/* xsquig.c - display squiggley tubular patterns in the X11 root window
**
** Copyright (C) 1990,1992,1993 by Jef Poskanzer <jef@acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdio.h>
#include <math.h>
#include <errno.h>
#include <malloc.h>
#include <signal.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#ifdef SVR4
#define SYSV
#endif

#ifdef SYSV
#include <sys/termio.h>
#endif /*SYSV*/

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xresource.h>
#define XTSTRINGDEFINES
#include <X11/StringDefs.h>


/* Definitions. */

#define X_CLASS "Xsquig"

#define DEFAULT_MAX_COLORS 256
#define MIN_COLORS 4
#define DEFAULT_LOOPS_PER_SECOND 20
#define INITIAL_LOOPS_PER_SLEEP 5
#define MIN_LOOPS_PER_SLEEP 1
#define INITIAL_SLEEP_USECS 75000
#define MIN_SLEEP_USECS 50000

#define NUM_BRANCHES 5
# define BRANCH_CIRCLE_G 0
# define BRANCH_CIRCLE_S 1
# define BRANCH_FOURSTAR 2
# define BRANCH_FIVESTAR 3
# define BRANCH_BOB 4

#define NOMINAL_RADIUS 40
#define MAX_RADIUS 60

#define MAX_POINTS_PER_OBJECT 3000

#define EXPOSE_CYCLES 30

#define SIZE_CYCLES 50

#define COLOR_CYCLES 40
#define MAX_D_COLOR 1000

#define MAX_D_COLOR_OFFSET 0.2

#define NUM_SINCOS 3000

#define abs(x) ((x) < 0 ? -(x) : (x))


/* Externals. */

extern char* getenv();
extern long random();
extern time_t time();


/* Forward routines. */

static void x_init();
static void x_alloc_colors();
static void x_cleanup();
static Window VirtualRootWindowOfScreen();
static void x_rdb_init();
static char* x_get_resource();
static int x_str_to_bool();
static void init_save();
static void save_point();
static void save_object();
static void line();
static void spline3();
static void init_sincos();
static void measure_circle();
static void measure_polygon();
static void measure_fourstar();
static void measure_fivestar();
static void measure_bob();
static void measure_objects();
static void stealth();
static void sigcatch();
static void init_squig();
static void main_loop();
static void init_timing();
static void timing();
static void init_colors();
static void store_colors();
static void object_drawproc();
static void draw_object();
static char* my_malloc();
#ifdef SYSV
static void usleep();
#endif /*SYSV*/


/* Globals. */

static char* argv0;
static char* app_name;
static int is_color;
static int max_colors;
static int loops_per_second;

static int loops_per_sleep;
static int sleep_usecs;

struct point {
    short x, y;
    u_char color;
    };

struct object {
    int xoff, yoff, w, h;
    Pixmap image;
    Pixmap mask;
    };

struct branch {
    int nobjects;
    struct object* objects;
    };

static struct branch branches[NUM_BRANCHES];
static int branch_index, object_index;
static int d_object_index;
static int object_clock;

static struct point points[MAX_POINTS_PER_OBJECT];
static int npoints;
static int minx, maxx, miny, maxy;

static XColor xcolors[DEFAULT_MAX_COLORS];
static unsigned short cred[3], cgrn[3], cblu[3];
static int dred[3], dgrn[3], dblu[3];
static float color_offset, d_color_offset;
static int i_color_offset, prev_i_color_offset;


/* Routines. */

void
main( argc, argv )
    int argc;
    char* argv[];
    {
    char* rval;
    char* usage = "usage: %s [-display d] [-maxcolors n] [-loops n] [-id]\n";

    argv0 = argv[0];
    app_name = "xsquig";
    max_colors = DEFAULT_MAX_COLORS;
    loops_per_second = DEFAULT_LOOPS_PER_SECOND;
    loops_per_sleep = INITIAL_LOOPS_PER_SLEEP;
    sleep_usecs = INITIAL_SLEEP_USECS;

    /* Parse args and initialize X stuff. */
    x_init( &argc, argv );

    /* Check usage. */
    if ( argc != 1 )
	{
	(void) fprintf( stderr, usage, argv[0] );
	exit( 1 );
	}

    if ( is_color )
	{
	/* Check max colors. */
	rval = x_get_resource( "maxcolors", "Maxcolors" );
	if ( rval != (char*) 0 )
	    max_colors = atoi(rval);
	x_alloc_colors();
	}

    /* Check loops. */
    rval = x_get_resource( "loops", "Loops" );
    if ( rval != (char*) 0 )
	loops_per_second = atoi(rval);

    /* Fork, if necessary. */
    rval = x_get_resource( "id", "Id" );
    if ( rval != (char*) 0 )
	if ( x_str_to_bool( rval ) )
	    stealth();

    /* Initialize. */
    init_squig();

    /* Main loop. */
    main_loop();
    
    /* Done. */
    x_cleanup();
    exit( 0 );
    }


/* X stuff. */

static Display* display = (Display*) 0;
static int screennum;
static Screen* screen;
static Window root;
static int width, height;
static int depth;
static unsigned long black, white;
static GC blackgc;
static GC imagegc;
static GC maskgc;
static Colormap cmap;
static int ncolors, ncolorso3;
static float fncolorso3;
static unsigned long pixels[DEFAULT_MAX_COLORS];

static void
x_init( argcP, argv )
    int* argcP;
    char** argv;
    {
    char* display_name;
    char* rval;
    int i, d;
    Pixmap onepixmap;

    /* Scan args looking for -display. */
    display_name = (char*) 0;
    for ( i = 1; i + 1 < *argcP; ++i )
	{
	if ( strcmp( argv[i], "-display" ) == 0 ||
	     strcmp( argv[i], "-displa" ) == 0 ||
	     strcmp( argv[i], "-displ" ) == 0 ||
	     strcmp( argv[i], "-disp" ) == 0 ||
	     strcmp( argv[i], "-dis" ) == 0 ||
	     strcmp( argv[i], "-di" ) == 0 ||
	     strcmp( argv[i], "-d" ) == 0 )
	    {
	    display_name = argv[i + 1];
	    for ( i = i + 2; i <= *argcP; ++i )
		argv[i - 2] = argv[i];
	    *argcP -= 2;
	    break;
	    }
	}

    display = XOpenDisplay( display_name );
    if ( display == (Display*) 0 )
	{
	(void) fprintf(
	    stderr, "%s: can't open display \"%s\"\n", argv0,
	    XDisplayName( display_name ) );
	exit( 1 );
	}

    screennum = DefaultScreen( display );
    screen = ScreenOfDisplay( display, DefaultScreen( display ) );
    root = VirtualRootWindowOfScreen( screen );
    width = WidthOfScreen( screen );
    height = HeightOfScreen( screen );
    depth = DefaultDepthOfScreen( screen );
    if ( depth <= 1 )
	{
	(void) fprintf( stderr, "(Using worthless black & white mode.)\n" );
	is_color = 0;
	}
    else
	is_color = 1;
    black = BlackPixelOfScreen( screen );
    white = WhitePixelOfScreen( screen );
    blackgc = XCreateGC( display, root, 0, (XGCValues*) 0 );
    XSetGraphicsExposures( display, blackgc, False );
    XSetForeground( display, blackgc, black );
    imagegc = XCreateGC( display, root, 0, (XGCValues*) 0 );
    XSetGraphicsExposures( display, imagegc, False );
    onepixmap = XCreatePixmap( display, root, 1, 1, 1 );
    maskgc = XCreateGC( display, onepixmap, 0, (XGCValues*) 0 );
    XFreePixmap( display, onepixmap );
    XSetGraphicsExposures( display, maskgc, False );
    cmap = DefaultColormapOfScreen( screen );

    x_rdb_init( argcP, argv );

    rval = x_get_resource( XtNname, "Name" );
    if ( rval != (char*) 0 )
	app_name = rval;

    rval = x_get_resource( "synchronous", "Synchronous" );
    if ( rval != (char*) 0 )
	if ( x_str_to_bool( rval ) )
	    XSynchronize( display, True );
    }

static void
x_alloc_colors()
    {
    for ( ncolors = max_colors; ncolors >= MIN_COLORS; --ncolors )
	if ( XAllocColorCells(
		display, cmap, False, (unsigned long*) 0, 0, pixels, ncolors ) )
	    break;
    if ( ncolors < MIN_COLORS )
	{
	(void) fprintf( stderr, "%s: can't allocate enough colors\n", argv0 );
	exit( 1 );
	}
    if ( ncolors < max_colors )
	(void) fprintf(
	    stderr, "%s: only %d colors available\n", argv0, ncolors );
    ncolorso3 = ncolors / 3;
    fncolorso3 = ncolors / 3.0;
    }

static void
x_cleanup()
    {
    XClearArea( display, root, 0, 0, width, height, True );
    if ( is_color )
	XFreeColors( display, cmap, pixels, ncolors, (unsigned long) 0 );
    XFreeGC( display, blackgc );
    XFreeGC( display, imagegc );
    XFreeGC( display, maskgc );
    XCloseDisplay( display );
    }


/* From vroot.h by Andreas Stolcke. */

static Window
VirtualRootWindowOfScreen( screenP )
    Screen* screenP;
    {
    static Window root = (Window) 0;
    Display* dpy = DisplayOfScreen( screenP );
    Atom __SWM_VROOT = None;
    int i;
    Window rootReturn, parentReturn;
    Window* children;
    unsigned int numChildren;

    root = RootWindowOfScreen( screenP );

    /* Go look for a virtual root. */
    __SWM_VROOT = XInternAtom( dpy, "__SWM_VROOT", False );
    if ( XQueryTree(
	     dpy, root, &rootReturn, &parentReturn, &children,
	     &numChildren ) )
	{
	for ( i = 0; i < numChildren; ++i )
	    {
	    Atom actual_type;
	    int actual_format;
	    unsigned long nitems, bytesafter;
	    Window* newRoot = (Window*) 0;

	    if ( XGetWindowProperty(
		     dpy, children[i], __SWM_VROOT, 0, 1, False, XA_WINDOW,
		     &actual_type, &actual_format, &nitems, &bytesafter,
		     (unsigned char**) &newRoot ) == Success && newRoot )
		{
		root = *newRoot;
		break;
		}
	    }
	if ( children )
	    XFree( (char*) children );
	}

    return root;
    }


/* X resources stuff. */

static XrmDatabase rdb;

static XrmOptionDescRec x_options[] = {
    { "-maxcolors",      "*maxcolors",      XrmoptionSepArg, (caddr_t) 0 },
    { "-loops",          "*loops",          XrmoptionSepArg, (caddr_t) 0 },
    { "-id",             "*id",             XrmoptionNoArg,  (caddr_t) "on" },
    { "-name",           ".name",           XrmoptionSepArg, (caddr_t) 0 },
    { "-synchronous",    "*synchronous",    XrmoptionNoArg,  (caddr_t) "on" },
    { "-xrm",            (char*) 0,         XrmoptionResArg, (caddr_t) 0 },
    };

static void
x_rdb_init( argcP, argv )
    int* argcP;
    char** argv;
    {
    char* resource_string;
    char* xenv;
    XrmDatabase xenv_rdb;

    XrmInitialize();

    /* Look for resource databases on server. */
    resource_string = XResourceManagerString( display );
    if ( resource_string != (char*) 0 )
	rdb = XrmGetStringDatabase( resource_string );
    else
	{
	/* No server databases, try ~/.Xdefaults */
	char* cp;
	char buf[500];

	cp = getenv( "HOME" );
	if ( cp != (char*) 0 )
	    (void) strcpy( buf, cp );
	else
	    {
	    struct passwd* pw;

	    cp = getenv( "USER" );
	    if ( cp != (char*) 0 )
		pw = getpwnam( cp );
	    else
		pw = getpwuid( getuid() );
	    if ( pw != (struct passwd*) 0 )
		(void) strcpy( buf, pw->pw_dir );
	    else
		(void) strcpy( buf, "." );	/* best we can do */
	    }
	(void) strcat( buf, "/.Xdefaults" );
	rdb = XrmGetFileDatabase( buf );
	}

    /* Merge in XENVIRONMENT, if any. */
    xenv = getenv( "XENVIRONMENT" );
    if ( xenv != (char*) 0 )
	{
	xenv_rdb = XrmGetFileDatabase( xenv );
	XrmMergeDatabases( xenv_rdb, &rdb );
	}

    /* And add command line options. */
    XrmParseCommand(
	&rdb, x_options, sizeof(x_options) / sizeof(*x_options),
	app_name, argcP, argv );
    }

static char*
x_get_resource( name, class )
    char* name;
    char* class;
    {
    char rname[500], rclass[500];
    char* type;
    XrmValue value;

    (void) sprintf( rname, "%s.%s", app_name, name );
    (void) sprintf( rclass, "%s.%s", X_CLASS, class );
    if ( XrmGetResource( rdb, rname, rclass, &type, &value ) == True )
	if ( strcmp( type, XtRString ) == 0 )
	    return (char*) value.addr;
    return (char*) 0;
    }

static int
x_str_to_bool( str )
    char* str;
    {
    if ( strcmp( str, "True" ) == 0 ||
         strcmp( str, "true" ) == 0 ||
         strcmp( str, "Yes" ) == 0 ||
         strcmp( str, "yes" ) == 0 ||
         strcmp( str, "On" ) == 0 ||
         strcmp( str, "on" ) == 0 ||
         strcmp( str, "Si" ) == 0 ||
         strcmp( str, "si" ) == 0 ||
         strcmp( str, "Da" ) == 0 ||
         strcmp( str, "da" ) == 0 ||
         strcmp( str, "T" ) == 0 ||
         strcmp( str, "t" ) == 0 ||
         strcmp( str, "Y" ) == 0 ||
         strcmp( str, "y" ) == 0 ||
         strcmp( str, "1" ) == 0 )
	return True;
    return False;
    }


/* Object measuring routines. */

static void
init_save()
    {
    npoints = 0;
    minx = miny = 32767;
    maxx = maxy = -32767;
    }

static void
save_point( x, y )
    int x, y;
    {
    static int prevx, prevy;

    y = -y;	/* coordinate system screwup, oh well */
    if ( npoints == 0 || x != prevx || y != prevy )
	{
	prevx = x;
	prevy = y;
	if ( npoints >= MAX_POINTS_PER_OBJECT )
	    {
	    (void) fprintf(
		stderr, "too many points in object %d,%d\n",
		branch_index, object_index );
	    exit( 1 );
	    }
	points[npoints].x = x;
	points[npoints].y = y;
	if ( x < minx ) minx = x;
	if ( x > maxx ) maxx = x;
	if ( y < miny ) miny = y;
	if ( y > maxy ) maxy = y;
	++npoints;
	}
    }

static void
save_object()
    {
    int w, h, i, c, x, y;
    Pixmap image, mask;

    branches[branch_index].objects[object_index].xoff = minx;
    branches[branch_index].objects[object_index].yoff = miny;
    w = maxx - minx + 1;
    h = maxy - miny + 1;
    branches[branch_index].objects[object_index].w = w;
    branches[branch_index].objects[object_index].h = h;
    image = XCreatePixmap( display, root, w, h, depth );
    branches[branch_index].objects[object_index].image = image;
    mask = XCreatePixmap( display, root, w, h, 1 );
    branches[branch_index].objects[object_index].mask = mask;
    XSetForeground( display, maskgc, 0 );
    XFillRectangle( display, mask, maskgc, 0, 0, w, h );
    XSetForeground( display, maskgc, 1 );

    c = 0;
    for ( i = 0; i < npoints; ++i )
	{
	if ( is_color )
	    XSetForeground( display, imagegc, pixels[ncolors * i / npoints] );
	else
	    {
	    XSetForeground( display, imagegc, ((c>>2)&1) ? white : black );
	    ++c;
	    }
	x = points[i].x - minx;
	y = points[i].y - miny;
	XDrawPoint( display, image, imagegc, x, y );
	XDrawPoint( display, mask, maskgc, x, y );
	}
    }


/* Call-back DDAs taken from libppm. */

#define DDA_SCALE 8192

static void
line( x0, y0, x1, y1, drawprocP )
    int x0, y0, x1, y1;
    void (*drawprocP)();
    {
    /* Special case zero-length lines. */
    if ( x0 == x1 && y0 == y1 )
	{
	(*drawprocP)( x0, y0 );
	return;
	}

    /* Draw, using a simple DDA. */
    if ( abs( x1 - x0 ) > abs( y1 - y0 ) )
	{ /* Loop over X domain. */
	register long dy, srow;
	register int dx, col, row, prevrow;

	if ( x1 > x0 )
	    dx = 1;
	else
	    dx = -1;
	dy = ( y1 - y0 ) * DDA_SCALE / abs( x1 - x0 );
	prevrow = row = y0;
	srow = row * DDA_SCALE + DDA_SCALE / 2;
	col = x0;
	for (;;)
	    {
	    if ( row != prevrow )
		{
		(*drawprocP)( col, prevrow );
		prevrow = row;
		}
	    (*drawprocP)( col, row );
	    if ( col == x1 )
		break;
	    srow += dy;
	    row = srow / DDA_SCALE;
	    col += dx;
	    }
	}
    else
	{ /* Loop over Y domain. */
	register long dx, scol;
	register int dy, col, row, prevcol;

	if ( y1 > y0 )
	    dy = 1;
	else
	    dy = -1;
	dx = ( x1 - x0 ) * DDA_SCALE / abs( y1 - y0 );
	row = y0;
	prevcol = col = x0;
	scol = col * DDA_SCALE + DDA_SCALE / 2;
	for (;;)
	    {
	    if ( col != prevcol )
		{
		(*drawprocP)( prevcol, row );
		prevcol = col;
		}
	    (*drawprocP)( col, row );
	    if ( row == y1 )
		break;
	    row += dy;
	    scol += dx;
	    col = scol / DDA_SCALE;
	    }
	}
    }

#define SPLINE_THRESH 3
static void
spline3( x0, y0, x1, y1, x2, y2, drawprocP )
    int x0, y0, x1, y1, x2, y2;
    void (*drawprocP)();
    {
    register int xa, ya, xb, yb, xc, yc, xp, yp;

    xa = ( x0 + x1 ) / 2;
    ya = ( y0 + y1 ) / 2;
    xc = ( x1 + x2 ) / 2;
    yc = ( y1 + y2 ) / 2;
    xb = ( xa + xc ) / 2;
    yb = ( ya + yc ) / 2;

    xp = ( x0 + xb ) / 2;
    yp = ( y0 + yb ) / 2;
    if ( abs( xa - xp ) + abs( ya - yp ) > SPLINE_THRESH )
	spline3( x0, y0, xa, ya, xb, yb, drawprocP );
    else
	line( x0, y0, xb, yb, drawprocP );

    xp = ( x2 + xb ) / 2;
    yp = ( y2 + yb ) / 2;
    if ( abs( xc - xp ) + abs( yc - yp ) > SPLINE_THRESH )
	spline3( xb, yb, xc, yc, x2, y2, drawprocP );
    else
	line( xb, yb, x2, y2, drawprocP );
    }


/* More object measuring routines. */

static float sins[NUM_SINCOS], coss[NUM_SINCOS];

static void
init_sincos()
    {
    int i;
    float f, df;

    df = 2.0 * M_PI / NUM_SINCOS;
    for ( i = 0, f = 0.0; i < NUM_SINCOS; ++i, f += df )
	{
	sins[i] = sin( f );
	coss[i] = cos( f );
	}
    }

static void
measure_circle( type )
    int type;
    {
    int count, i;
    float r, dr;

    branch_index = type;
    switch ( type )
	{
	case BRANCH_CIRCLE_G:
	count = ( MAX_RADIUS - NOMINAL_RADIUS ) * 2 + 1;
	dr = 0.5;
	break;

	case BRANCH_CIRCLE_S:
	count = NOMINAL_RADIUS;
	dr = -0.5;
	break;
	}
    branches[branch_index].nobjects = count;
    branches[branch_index].objects =
	(struct object*) my_malloc( count * sizeof(struct object) );
    for ( object_index = 0, r = NOMINAL_RADIUS;
	  object_index < count; ++object_index, r += dr )
	{
	init_save();
	for ( i = 0; i < NUM_SINCOS; ++i )
	    save_point( (int) ( r * sins[i] ), (int) ( r * coss[i] ) );
	save_object();
	}
    }

static void
measure_polygon( bi, n, xs, ys, count )
    int bi, n, count;
    float* xs;
    float* ys;
    {
    int i, l;
    float r, u, up, v, vp, x, xp, y, yp;

    branch_index = bi;
    branches[branch_index].nobjects = count;
    branches[branch_index].objects =
	(struct object*) my_malloc( count * sizeof(struct object) );
    r = NOMINAL_RADIUS;
    for ( object_index = 0; object_index < count; ++object_index )
	{
	up = (float) object_index / count;
	u = 1.0 - up;
	init_save();
	for ( i = 0; i < NUM_SINCOS; ++i )
	    {
	    l = i * n / NUM_SINCOS;
	    vp = (float) ( i * n % NUM_SINCOS ) / NUM_SINCOS;
	    v = 1.0 - vp;
	    xp = v *  xs[l] + vp *  xs[(l+1) % n];
	    yp = v *  ys[l] + vp *  ys[(l+1) % n];
	    x = r * ( u * sins[i] + 1.5 * up * xp );
	    y = r * ( u * coss[i] + 1.5 * up * yp );
	    save_point( (int) x, (int) y );
	    }
	save_object();
	}
    }

static void
measure_fourstar()
    {
    static float xs[] = {
	0.0, 0.2, 1.0, 0.2, 0.0, -0.2, -1.0, -0.2 };
    static float ys[] = {
	1.0, 0.2, 0.0, -0.2, -1.0, -0.2, 0.0, 0.2 };

    measure_polygon( BRANCH_FOURSTAR, sizeof(xs) / sizeof(*xs), xs, ys, 40 );
    }

static void
measure_fivestar()
    {
    static float xs[] = {
	0.0000, 0.2245, 0.9511, 0.3633, 0.5878,
	0.0000, -0.5878, -0.3633, -0.9511, -0.2245 };
    static float ys[] = {
	1.0000, 0.3090, 0.3090, -0.1180, -0.8090,
	-0.3820, -0.8090, -0.1180, 0.3090, 0.3090 };

    measure_polygon( BRANCH_FIVESTAR, sizeof(xs) / sizeof(*xs), xs, ys, 40 );
    }

static void
measure_bob()
    {
    static float xs[] = {
	0.0000, 0.1428, 0.2619, 0.3809, 0.5000,
	0.5952, 0.6905, 0.7143, 0.7143, 0.7143,
	0.7381, 0.7143, 0.6905, 0.6667, 0.6428,
	0.5476, 0.5238, 0.5000, 0.4524, 0.3571,
	0.2857, 0.1905, 0.0714, -0.0476, -0.1905,
	-0.2857, -0.4047, -0.5000, -0.5952, -0.7143,
	-0.7857, -0.8095, -0.7143, -0.5952, -0.4761,
	-0.3809, -0.4524, -0.5000, -0.5238, -0.5714,
	-0.6428, -0.6667, -0.6905, -0.6905, -0.6667,
	-0.6905, -0.6905, -0.6667, -0.6428, -0.5714,
	-0.476, -0.3571, -0.3095, -0.1429 };
    static float ys[] = {
	1.0000, 1.0000, 0.9762, 0.9524, 0.8571,
	0.7381, 0.6428, 0.5238, 0.3809, 0.2381,
	0.1190, 0.0000, -0.1190, -0.2381, -0.3571,
	-0.4762, -0.5952, -0.7143, -0.8333, -0.9524,
	-1.0714, -1.1666, -1.1904, -1.1666, -1.1666,
	-1.0714, -1.1190, -1.2381, -1.3333, -1.3095,
	-1.1904, -1.0714, -0.9524, -0.9762, -1.0238,
	-0.9286, -0.8095, -0.6905, -0.5714, -0.4524,
	-0.3333, -0.2143, -0.0952, 0.0476, 0.1667,
	0.2857, 0.4286, 0.5476, 0.6667, 0.7857,
	0.9047, 0.9524, 0.9762, 1.0000 };

    measure_polygon( BRANCH_BOB, sizeof(xs) / sizeof(*xs), xs, ys, 40 );
    }

static void
measure_objects()
    {
    (void) fprintf( stderr, "Initializing..." );  (void) fflush( stderr );
    init_sincos();
    (void) fprintf( stderr, "." );  (void) fflush( stderr );
    measure_circle( BRANCH_CIRCLE_S );
    (void) fprintf( stderr, "." );  (void) fflush( stderr );
    measure_circle( BRANCH_CIRCLE_G );
    (void) fprintf( stderr, "." );  (void) fflush( stderr );
    measure_fourstar();
    (void) fprintf( stderr, "." );  (void) fflush( stderr );
    measure_fivestar();
    (void) fprintf( stderr, "." );  (void) fflush( stderr );
    measure_bob();
    (void) fprintf( stderr, " done.\n" );
    }


/* Generic application stuff. */

static void
stealth()
    {
    int pid, tty;

    pid = fork();
    if ( pid < 0 )
	{
	perror( "fork" );
	exit( 1 );
	}
    else if ( pid > 0 )
	/* Parent just exits. */
	exit( 0 );
    (void) printf( "%d\n", getpid() );
    (void) fflush( stdout );

    /* Go stealth (ditch our controlling tty). */
    tty = open( "/dev/tty", 0 );
    if ( tty < 0 )
	{
	/* ENXIO means that there is no controlling terminal, so we don't
	** have to detach anything.
	*/
        if ( errno != ENXIO )
	    {
	    (void) fprintf( stderr, "%s: ", argv0 );
	    perror( "/dev/tty open" );
	    exit( 1 );
	    }
	}
    else
	{
#ifdef TIOCNOTTY
	if ( ioctl( tty, TIOCNOTTY, 0 ) < 0 )
	    {
	    (void) fprintf( stderr, "%s: ", argv0 );
	    perror( "TIOCNOTTY ioctl" );
	    exit( 1 );
	    }
#endif /*TIOCNOTTY*/
	(void) close( tty );
	}
    }

static int goflag;
static int loops;

static void
sigcatch()
    {
    goflag = 0;
    }

static void
init_squig()
    {
    /* Initialize the random number generator. */
    srandom( (int) ( time( (long*) 0 ) ^ getpid() ) );

    /* Set up for signal catching. */
    goflag = 1;
    (void) signal( SIGHUP, sigcatch );
    (void) signal( SIGINT, sigcatch );
    (void) signal( SIGTERM, sigcatch );
    (void) signal( SIGFPE, sigcatch );

    /* Initialize the objects. */
    measure_objects();

    if ( is_color )
	{
	/* Initialize the colors. */
	init_colors();
	}

    /* Clear the window. */
    XSelectInput( display, root, ExposureMask );
    XClearArea( display, root, 0, 0, width, height, True );

    /* Initialize the timing. */
    init_timing();
    }

static void
main_loop()
    {
    int thiscx, thiscy, nextcx, nextcy, prevex, prevey, nextex, nextey;

    /* Initialize spline points. */
    thiscx = random() % ( width - 2 * MAX_RADIUS ) + MAX_RADIUS;
    thiscy = random() % ( height - 2 * MAX_RADIUS ) + MAX_RADIUS;
    nextcx = random() % ( width - 2 * MAX_RADIUS ) + MAX_RADIUS;
    nextcy = random() % ( height - 2 * MAX_RADIUS ) + MAX_RADIUS;
    nextex = ( nextcx + thiscx ) / 2;
    nextey = ( nextcy + thiscy ) / 2;

    branch_index = BRANCH_CIRCLE_G;
    object_index = 0;
    d_object_index = 1;
    object_clock = 0;

    while ( goflag )
	{
	thiscx = nextcx;
	thiscy = nextcy;
	nextcx = random() % ( width - 2 * MAX_RADIUS ) + MAX_RADIUS;
	nextcy = random() % ( height - 2 * MAX_RADIUS ) + MAX_RADIUS;
	prevex = nextex;
	prevey = nextey;
	nextex = ( nextcx + thiscx ) / 2;
	nextey = ( nextcy + thiscy ) / 2;
	spline3(
	    prevex, prevey, thiscx, thiscy, nextex, nextey,
	    object_drawproc );
	}
    }


/* Timing stuff. */

static int then_loops;
static time_t then;

static void
init_timing()
    {
    then = time( (time_t*) 0 );
    then_loops = 0;
    loops = 1;
    }

static void
timing()
    {
    float ratio;
    int t;
    time_t now;

    ++loops;
    if ( sleep_usecs > 0 && loops_per_sleep > 0 &&
	 loops % loops_per_sleep == 0 )
	{
	XFlush( display );
	usleep( sleep_usecs );
	if ( loops_per_second > 0 )
	    {
	    now = time( (time_t*) 0 );
	    if ( now != then )
		{
		if ( then_loops != 0 )
		    {
		    ratio = (float) ( loops - then_loops ) /
			    (float) loops_per_second;
		    if ( ratio > 1.0 )
			{
			/* We're going too fast. */
			t = loops_per_sleep / ratio;
			if ( t >= MIN_LOOPS_PER_SLEEP )
			    loops_per_sleep = t;
			else
			    sleep_usecs *= ratio;
			}
		    else if ( ratio < 1.0 )
			{
			/* We're going too slow. */
			t = sleep_usecs * ratio;
			if ( t >= MIN_SLEEP_USECS )
			    sleep_usecs = t;
			else
			    {
			    loops_per_sleep /= ratio;
			    if ( loops_per_sleep > 3 * loops_per_second )
				loops_per_sleep = 0;	/* give up */
			    }
			}
		    }
		then_loops = loops;
		then = now;
		}
	    }
	}
    }


/* Squig smarts. */

static void
init_colors()
    {
    int i;

    for ( i = 0; i < ncolors; ++i )
	{
	xcolors[i].red = xcolors[i].green = xcolors[i].blue = 0;
	xcolors[i].flags = DoRed|DoGreen|DoBlue;
	}
    for ( i = 0; i < 3; ++i )
	{
	cred[i] = cgrn[i] = cblu[i] = 0;
	dred[i] = random() % ( MAX_D_COLOR - 1 ) + 1;
	dgrn[i] = random() % ( MAX_D_COLOR - 1 ) + 1;
	dblu[i] = random() % ( MAX_D_COLOR - 1 ) + 1;
	}
    color_offset = 0.0;
    d_color_offset = 0.0;
    i_color_offset = prev_i_color_offset = 0;
    store_colors();
    }

static void
store_colors()
    {
    int i;

    for ( i = 0; i < ncolors; ++i )
	xcolors[i].pixel = pixels[(i + i_color_offset ) % ncolors];
    XStoreColors( display, cmap, xcolors, ncolors );
    }

static void
draw_object( x, y )
    int x, y;
    {
    register struct object* op;
    register int ox, oy;

    op = &(branches[branch_index].objects[object_index]);
    ox = x + op->xoff;
    oy = y + op->yoff;
    XSetClipMask( display, imagegc, op->mask );
    XSetClipOrigin( display, imagegc, ox, oy );
    XCopyArea(
	display, op->image, root, imagegc, 0, 0, op->w, op->h, ox, oy );
    }

static void
object_drawproc( x, y )
    int x, y;
    {
    register unsigned long r;

    if ( ! goflag )
	return;

    ++object_clock;
    r = random() >> 2;

    /* Handle X expose events? */
    if ( object_clock % EXPOSE_CYCLES == 1 )
	{
	if ( XPending( display ) != 0 )
	    {
	    XEvent ev;

	    XNextEvent( display, &ev );
	    if ( ev.type == Expose )
		XFillRectangle(
		    display, root, blackgc, ev.xexpose.x, ev.xexpose.y,
		    ev.xexpose.width, ev.xexpose.height );
	    }
	}

    /* Adjust object? */
    if ( object_clock % SIZE_CYCLES == 0 )
	{
	register int i;

	/* First draw old object with new position, to avoid gaps. */
	draw_object( x, y );

	/* Advance object index, changing delta if necessary or random. */
	if ( ( r % 46301 ) % 20 == 0 )
	    d_object_index = ( random() >> 2 ) % 3 - 1;
	i = object_index + d_object_index;
	if ( i < 0 )
	    {
	    do
		{
		d_object_index = ( random() >> 2 ) % 3 - 1;
		i = object_index + d_object_index;
		}
	    while ( i < 0 );
	    }
	else if ( i >= branches[branch_index].nobjects )
	    {
	    if ( ( r % 46307 ) % 2 == 0 )
		{
		i = branches[branch_index].nobjects - 1;
		d_object_index = 0;
		}
	    else
		{
		do
		    {
		    d_object_index = ( random() >> 2 ) % 3 - 1;
		    i = object_index + d_object_index;
		    }
		while ( i >= branches[branch_index].nobjects );
		}
	    }
	object_index = i;

	/* Switch branches? */
	if ( object_index == 0 )
	    switch ( ( random() >> 2 ) % 12 )
		{
		case  0:
		case  1:
		case  2: branch_index = BRANCH_CIRCLE_G; break;

		case  3:
		case  4:
		case  5: branch_index = BRANCH_CIRCLE_S; break;

		case  6:
		case  7: branch_index = BRANCH_FOURSTAR; break;

		case  8:
		case  9: branch_index = BRANCH_FIVESTAR; break;

		case 10:
		case 11: branch_index = BRANCH_BOB; break;
		}
	}

    if ( is_color )
	{
	int newcolormap = 0;

	/* Adjust color offset. */
	if ( r % 60 == 0 )
	    {
	    register float f;

	    for (;;)
		{
		f = d_color_offset + ( (random()>>2) % 2001 - 1000 ) / 30000.0;
		if ( f <= MAX_D_COLOR_OFFSET && f >= -MAX_D_COLOR_OFFSET )
		    break;
		}
	    d_color_offset = f;
	    }
	color_offset += d_color_offset;
	/* Got to be careful to stay positive, since % is NOT modulo. */
	if ( color_offset < 0.0 )
	    color_offset += ncolors;
	i_color_offset = color_offset;
	if ( i_color_offset != prev_i_color_offset )
	    {
	    newcolormap = 1;
	    prev_i_color_offset = i_color_offset;
	    }

	/* Adjust colors. */
	if ( object_clock % COLOR_CYCLES == 0 )
	    {
	    register int i;
	    register long j;

	    for ( i = 0; i < 3; ++i )
		{
		for (;;)
		    {
		    j = cred[i] + dred[i];
		    if ( j >= 0 && j <= 65535 ) break;
		    dred[i] = (random()>>2) % ( MAX_D_COLOR * 2 ) - MAX_D_COLOR;
		    if ( dred[i] <= 0 ) --dred[i];
		    }
		cred[i] = j;
		for (;;)
		    {
		    j = cgrn[i] + dgrn[i];
		    if ( j >= 0 && j <= 65535 ) break;
		    dgrn[i] = (random()>>2) % ( MAX_D_COLOR * 2 ) - MAX_D_COLOR;
		    if ( dgrn[i] <= 0 ) --dgrn[i];
		    }
		cgrn[i] = j;
		for (;;)
		    {
		    j = cblu[i] + dblu[i];
		    if ( j >= 0 && j <= 65535 ) break;
		    dblu[i] = (random()>>2) % ( MAX_D_COLOR * 2 ) - MAX_D_COLOR;
		    if ( dblu[i] <= 0 ) --dblu[i];
		    }
		cblu[i] = j;
		}

	    for ( i = 0; i < ncolors; ++i )
		if ( i < ncolorso3 )
		    {
		    xcolors[i].red = cred[0] + ( cred[1] - cred[0] ) *
			i / fncolorso3;
		    xcolors[i].blue = cblu[0] + ( cblu[1] - cblu[0] ) *
			i / fncolorso3;
		    xcolors[i].green = cgrn[0] + ( cgrn[1] - cgrn[0] ) *
			i / fncolorso3;
		    }
		else if ( i < 2 * ncolorso3 )
		    {
		    xcolors[i].red = cred[1] + ( cred[2] - cred[1] ) *
			( i - ncolorso3 ) / fncolorso3;
		    xcolors[i].blue = cblu[1] + ( cblu[2] - cblu[1] ) *
			( i - ncolorso3 ) / fncolorso3;
		    xcolors[i].green = cgrn[1] + ( cgrn[2] - cgrn[1] ) *
			( i - ncolorso3 ) / fncolorso3;
		    }
		else
		    {
		    xcolors[i].red = cred[2] + ( cred[0] - cred[2] ) *
			( i - 2 * ncolorso3 ) / fncolorso3;
		    xcolors[i].blue = cblu[2] + ( cblu[0] - cblu[2] ) *
			( i - 2 * ncolorso3 ) / fncolorso3;
		    xcolors[i].green = cgrn[2] + ( cgrn[0] - cgrn[2] ) *
			( i - 2 * ncolorso3 ) / fncolorso3;
		    }

	    newcolormap = 1;
	    }

	/* Store new colors. */
	if ( newcolormap )
	    store_colors();
	}

    /* And draw it. */
    draw_object( x, y );

    /* Do timing stuff. */
    timing();
    }


/* Utilities. */

static char*
my_malloc( size )
    int size;
    {
    char* p;

    p = (char*) malloc( (unsigned) size );
    if ( p == (char*) 0 )
	{
	(void) fprintf( stderr, "out of memory\n" );
	exit( 1 );
	}
    return p;
    }

#ifdef SYSV

/* Most SysV's don't have a usleep.  Some of them have select. */

static void
usleep( usecs )
int usecs;
    {
    struct timeval timeout;

    timeout.tv_sec = usecs / 1000000;
    timeout.tv_usec = usecs % 1000000;
    select( 0, 0, 0, 0, &timeout );
    }

#endif /*SYSV*/
