/* xbouncebits - displays moving bouncing bitmaps in the X11 root window
**
** Copyright (C) 1992 by Jef Poskanzer <jef@acme.com>.  All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdio.h>
#if defined(SYSV) || defined(SVR4)
#include <string.h>
#include <sys/termio.h>
#else /*SYSV*/
#include <strings.h>
#endif /*SYSV*/
#include <signal.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <errno.h>
extern char* sys_errlist[];	/* in case errno.h doesn't declare it */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xresource.h>
#define XTSTRINGDEFINES
#include <X11/StringDefs.h>
#include "Rects.h"

/* Add fd_set definitions, in case the system doesn't have them. */
#ifndef FD_SET
#define NFDBITS		32
#define FD_SETSIZE	32
#define FD_SET(n, p)	((p)->fds_bits[(n)/NFDBITS] |= (1 << ((n) % NFDBITS)))
#define FD_CLR(n, p)	((p)->fds_bits[(n)/NFDBITS] &= ~(1 << ((n) % NFDBITS)))
#define FD_ISSET(n, p)	((p)->fds_bits[(n)/NFDBITS] & (1 << ((n) % NFDBITS)))
#define FD_ZERO(p)	bzero((char *)(p), sizeof(*(p)))
#endif


/* Definitions. */

#define X_CLASS "Xbouncebits"
#define MAXDISPLAYS 30
#define MAXSCREENS 50

#define CYCLES 10			/* default cycles per second */
#define GRAVCONST 25			/* cycles per gravity acceleration */
#define INITIAL_JUMPS 10		/* # of jumps to randomize position */
#define COLLISION_TRYS 50		/* try this many times, then give up */

#define M_JUMP 1
#define M_GLIDE 2
#define M_GRAVITY 3

typedef struct rect_struct {
    XRectangle bb;
    Pixmap pixmap[MAXSCREENS];
    int mode;			/* M_JUMP, M_GLIDE, or M_GRAVITY */
    int clock;			/* for M_JUMP */
    int dx, dy;			/* for M_GLIDE and M_GRAVITY */
    } rect;

#define abs(x) ( (x) >= 0 ? (x) : -(x) )
#define max(a,b) ( (a) >= (b) ? (a) : (b) )


/* Externals. */

extern char* getenv();
extern char* malloc();
extern long random();
extern char* realloc();
extern long time();


/* Forward routines. */

static void add_rect();
static void make_pixmap();
static void x_init();
static void display_init();
static void screen_init();
static Window VirtualRootWindowOfScreen();
static void xrdb_init();
static char* x_get_resource();
static int x_str_to_bool();
static int x_get_color_resource();
static void x_cleanup();
static void reverse_screens();
static void stealth();
static void rects_place();
static void rect_bounce();
static void rect_sleep();
static void rect_glide();
static void rect_jump();
static void find_leaders();
static void main_loop();
static void sigcatch();
static int check_events();
static void handle_event();
static int screen_of_window();
static void expose();
static void paint_rect();

static void move_rects();
static int check_collisions();


/* Variables. */

static char* argv0;

static int nrects;
static int maxrects;
static rect* rects;
static int cycles;
static int goflag;
static int gravclock;
static Bool smaller_screens;

static char* app_name;
int ndisplays, nscreens;
static Display* displays[MAXDISPLAYS];
static int d_of_s[MAXSCREENS];
static Screen* xscreen[MAXSCREENS];
static XrmDatabase rdb[MAXSCREENS];
static Window root[MAXSCREENS];
static XRectangle rootrect[MAXSCREENS];
static int depth[MAXSCREENS];
static unsigned long foreground[MAXSCREENS], background[MAXSCREENS];
static GC rectgc[MAXSCREENS];

static XRectangle allrect;

/* Routines. */

void
main( argc, argv )
    int argc;
    char* argv[];
    {
    char* rval;
    int mode, copies, i, j, s, width, height;
    char* usage = "usage: %s [-display d] ... [-rtol] [-cycles n] [-id] [-jump|-glide|-gravity|-copies n] bitmap ...\n";
    FILE* in;
    char* data;

    argv0 = argv[0];
    app_name = "xbouncebits";
    cycles = CYCLES;
    mode = M_JUMP;
    copies = 1;

    /* Parse args and initialize X stuff. */
    x_init( &argc, argv );
    RectsInit();

    /* Reverse right-for-left? */
    rval = x_get_resource( 0, "rtol", "Rtol" );
    if ( rval != (char*) 0 )
	if ( x_str_to_bool( rval ) )
	    reverse_screens();

    /* Change cycle time? */
    rval = x_get_resource( 0, "cycles", "Cycles" );
    if ( rval != (char*) 0 )
        {
        cycles = atoi( rval );
        if ( cycles < 0 )
	    {
	    (void) fprintf( stderr, usage, argv0 );
	    exit( 1 );
	    }
        }

    /* Check for any screens smaller than the rest. */
    smaller_screens = False;
    for ( s = 0; s < nscreens; ++s )
	if ( rootrect[s].height < allrect.height )
	    smaller_screens = True;

    /* Read in the bitmap flags and bitmaps. */
    nrects = 0;
    maxrects = 0;
    for ( i = 1; i < argc; ++i )
	{
	if ( strcmp( argv[i], "-jump" ) == 0 ||
	     strcmp( argv[i], "-jum" ) == 0 ||
	     strcmp( argv[i], "-ju" ) == 0 ||
	     strcmp( argv[i], "-j" ) == 0 )
	    mode = M_JUMP;
	else if ( strcmp( argv[i], "-glide" ) == 0 ||
	     strcmp( argv[i], "-glid" ) == 0 ||
	     strcmp( argv[i], "-gli" ) == 0 ||
	     strcmp( argv[i], "-gl" ) == 0 )
	    mode = M_GLIDE;
	else if ( strcmp( argv[i], "-gravity" ) == 0 ||
	     strcmp( argv[i], "-gravit" ) == 0 ||
	     strcmp( argv[i], "-gravi" ) == 0 ||
	     strcmp( argv[i], "-grav" ) == 0 ||
	     strcmp( argv[i], "-gra" ) == 0 ||
	     strcmp( argv[i], "-gr" ) == 0 )
	    mode = M_GRAVITY;
	else if ( strcmp( argv[i], "-copies" ) == 0 ||
	     strcmp( argv[i], "-copie" ) == 0 ||
	     strcmp( argv[i], "-copi" ) == 0 ||
	     strcmp( argv[i], "-cop" ) == 0 ||
	     strcmp( argv[i], "-co" ) == 0 ||
	     strcmp( argv[i], "-c" ) == 0 )
	    {
	    ++i;
	    copies = atoi( argv[i] );
	    if ( copies == 0 )
		{
		(void) fprintf( stderr, usage, argv0 );
		exit( 1 );
		}
	    }
	else if ( argv[i][0] == '-' && argv[i][1] != '\0' )
	    {
	    (void) fprintf( stderr, usage, argv0 );
	    exit( 1 );
	    }
	else
	    {
	    /* Read a bitmap. */
	    if ( strcmp( argv[i], "-" ) == 0 )
		in = stdin;
	    else
		{
		in = fopen( argv[i], "r" );
		if ( in == (FILE*) 0 )
		    {
		    (void) fprintf(
			stderr, "%s: %s - %s\n", argv0, argv[i],
			sys_errlist[errno] );
		    exit( 1 );
		    }
		}
	    j = XmuReadBitmapData(
		in, &width, &height, &data, (int*) 0, (int*) 0 );
	    if ( j != BitmapSuccess )
		{
		(void) fprintf(
		    stderr, "%s: error %d reading %s\n", argv0, j, argv[i] );
		exit( 1 );
		}
	    if ( in != stdin )
		fclose( in );
	    for ( j = 0; j < copies; ++j )
		{
		add_rect();
		rects[nrects].bb.width = (unsigned short) width;
		rects[nrects].bb.height = (unsigned short) height;
		for ( s = 0; s < nscreens; ++s )
		    make_pixmap( s, data, width, height );
		rects[nrects].mode = mode;
		++nrects;
		}
	    free( data );
	    }
	}
    if ( nrects == 0 )
	{
	(void) fprintf( stderr, usage, argv0 );
	exit( 1 );
	}

    /* Initialize the random number generator. */
    srandom( (int) ( time( (long*) 0 ) ^ getpid() ) );

    /* Position the rectangles. */
    rects_place();

    /* Fork, if necessary. */
    rval = x_get_resource( 0, "id", "Id" );
    if ( rval != (char*) 0 )
	if ( x_str_to_bool( rval ) )
	    stealth();

    /* Main loop. */
    main_loop();

    /*NOTREACHED*/
    }

static void
add_rect()
    {
    if ( nrects < maxrects )
	return;
    if ( maxrects == 0 )
	{
	maxrects = 10;		/* non-critical parameter */
	rects = (rect*) malloc( (unsigned) ( maxrects * sizeof(rect) ) );
	}
    else
	{
	maxrects *= 2;
	rects = (rect*) realloc(
	    (char*) rects, (unsigned) ( maxrects * sizeof(rect) ) );
	}
    if ( rects == (rect*) 0 )
	{
	(void) fprintf( stderr, "%s: out of memory\n", argv0 );
	exit( 1 );
	}
    }

static void
make_pixmap( s, data, width, height )
    int s, width, height;
    char* data;
    {
    int s2;

    for ( s2 = 0; s2 < s; ++s2 )
	{
	if ( d_of_s[s] == d_of_s[s2] && depth[s] == depth[s2] )
	    {
	    rects[nrects].pixmap[s] = rects[nrects].pixmap[s2];
	    return;
	    }
	}
    rects[nrects].pixmap[s] = XCreatePixmapFromBitmapData(
	displays[d_of_s[s]], root[s], data, width, height,
	foreground[s], background[s], depth[s] );
    }

static void
x_init( argcP, argv )
    int* argcP;
    char** argv;
    {
    char* rval;
    int i, d;

    /* Scan args looking for -display. */
    allrect.x = allrect.y = allrect.width = allrect.height = 0;
    ndisplays = nscreens = 0;
    try_again:
    for ( i = 1; i + 1 < *argcP; ++i )
	{
	if ( strcmp( argv[i], "-display" ) == 0 ||
	     strcmp( argv[i], "-displa" ) == 0 ||
	     strcmp( argv[i], "-displ" ) == 0 ||
	     strcmp( argv[i], "-disp" ) == 0 ||
	     strcmp( argv[i], "-dis" ) == 0 ||
	     strcmp( argv[i], "-di" ) == 0 ||
	     strcmp( argv[i], "-d" ) == 0 )
	    {
	    display_init( argv[i + 1], argcP, argv );
	    for ( i = i + 2; i <= *argcP; ++i )
		argv[i - 2] = argv[i];
	    *argcP -= 2;
	    goto try_again;
	    }
	}
    if ( nscreens == 0 )
	display_init( (char*) 0, argcP, argv );

    rval = x_get_resource( 0, XtNname, "Name" );
    if ( rval != (char*) 0 )
	app_name = rval;

    rval = x_get_resource( 0, "synchronous", "Synchronous" );
    if ( rval != (char*) 0 )
	if ( x_str_to_bool( rval ) )
	    for ( d = 0; d < ndisplays; ++d )
		XSynchronize( displays[d], True );
    }

static void
display_init( display_name, argcP, argv )
    char* display_name;
    int* argcP;
    char** argv;
    {
    Display* xdisplay;
    char* cp;
    int xscreennum;

    xdisplay = XOpenDisplay( display_name );
    if ( xdisplay == (Display*) 0 )
	{
	(void) fprintf(
	    stderr, "%s: can't open display \"%s\"\n", argv0,
	    XDisplayName( display_name ) );
	exit( 1 );
	}

    /* Check for specific screen in display_name. */
    if ( display_name != (char*) 0 )
	{
	cp = index( display_name, ':' );
	if ( cp != (char*) 0 )
	    {
	    if ( index( cp, '.' ) != (char*) 0 )
		{
		screen_init(
		    ndisplays, nscreens, xdisplay,
		    DefaultScreen( xdisplay ), argcP, argv );
		++nscreens;
		++ndisplays;
		return;
		}
	    }
	}

    /* No screen specified, use them all. */
    for ( xscreennum = 0; xscreennum < ScreenCount( xdisplay ); ++xscreennum )
	{
	screen_init( ndisplays, nscreens, xdisplay, xscreennum, argcP, argv );
	++nscreens;
	}
    ++ndisplays;
    }

static void
screen_init( d, s, xdisplay, xscreennum, argcP, argv )
    int d, s, xscreennum;
    Display* xdisplay;
    int* argcP;
    char** argv;
    {
    char* rval;
    int reverse_video;

    displays[d] = xdisplay;
    d_of_s[s] = d;
    xscreen[s] = ScreenOfDisplay( xdisplay, xscreennum );
    root[s] = VirtualRootWindowOfScreen( xscreen[s] );
    depth[s] = DefaultDepthOfScreen( xscreen[s] );
    rootrect[s].x = allrect.width;
    rootrect[s].y = 0;
    rootrect[s].width = WidthOfScreen( xscreen[s] );
    rootrect[s].height = HeightOfScreen( xscreen[s] );
    RectsUnion( &allrect, &rootrect[s] );

    xrdb_init( s, argcP, argv );

    reverse_video = False;
    rval = x_get_resource( s, XtNreverseVideo, XtCReverseVideo );
    if ( rval != (char*) 0 )
	reverse_video = x_str_to_bool( rval );

    if ( ! x_get_color_resource(
	       s, XtNforeground, XtCForeground, &foreground[s] ) )
	foreground[s] = BlackPixelOfScreen( xscreen[s] );

    if ( ! x_get_color_resource(
	       s, XtNbackground, XtCBackground, &background[s] ) )
	background[s] = WhitePixelOfScreen( xscreen[s] );

    if ( reverse_video )
	{
	unsigned long t;

	t = foreground[s];
	foreground[s] = background[s];
	background[s] = t;
	}

    rectgc[s] = XCreateGC( xdisplay, root[s], 0, (XGCValues*) 0 );
    XSetForeground( xdisplay, rectgc[s], foreground[s] );
    XSetBackground( xdisplay, rectgc[s], background[s] );
    }

/* From vroot.h by Andreas Stolcke. */
static Window
VirtualRootWindowOfScreen( screenP )
    Screen* screenP;
    {
    static Window root = (Window) 0;
    Display* dpy = DisplayOfScreen( screenP );
    Atom __SWM_VROOT = None;
    int i;
    Window rootReturn, parentReturn;
    Window* children;
    unsigned int numChildren;

    root = RootWindowOfScreen( screenP );

    /* Go look for a virtual root. */
    __SWM_VROOT = XInternAtom( dpy, "__SWM_VROOT", False );
    if ( XQueryTree(
	     dpy, root, &rootReturn, &parentReturn, &children,
	     &numChildren ) )
	{
	for ( i = 0; i < numChildren; ++i)
	    {
	    Atom actual_type;
	    int actual_format;
	    unsigned long nitems, bytesafter;
	    Window* newRoot = (Window*) 0;

	    if ( XGetWindowProperty(
		     dpy, children[i], __SWM_VROOT, 0, 1, False, XA_WINDOW,
		     &actual_type, &actual_format, &nitems, &bytesafter,
		     (unsigned char**) &newRoot ) == Success && newRoot )
		{
		root = *newRoot;
		break;
		}
	    }
	if ( children )
	    XFree( (char*) children );
	}

    return root;
    }

/* Resources stuff. */

static XrmOptionDescRec x_options[] = {
{"-background",        "*background",        XrmoptionSepArg, (caddr_t) 0},
{"-bg",                "*background",        XrmoptionSepArg, (caddr_t) 0},
{"-cycles",            "*cycles",            XrmoptionSepArg, (caddr_t) 0},
{"-fg",                "*foreground",        XrmoptionSepArg, (caddr_t) 0},
{"-foreground",        "*foreground",        XrmoptionSepArg, (caddr_t) 0},
{"-id",                "*id",                XrmoptionNoArg,  (caddr_t) "on"},
{"-name",              ".name",              XrmoptionSepArg, (caddr_t) 0},
{"-reverse",           "*reverseVideo",      XrmoptionNoArg,  (caddr_t) "on"},
{"-rtol",              "*rtol",              XrmoptionNoArg,  (caddr_t) "on"},
{"-rv",                "*reverseVideo",      XrmoptionNoArg,  (caddr_t) "on"},
{"-synchronous",       "*synchronous",       XrmoptionNoArg,  (caddr_t) "on"},
{"-xrm",               (char*) 0,            XrmoptionResArg, (caddr_t) 0},
};

static void
xrdb_init( s, argcP, argv )
    int s;
    int* argcP;
    char** argv;
    {
    char* resource_string;
    char* xenv;
    XrmDatabase xenv_rdb;

    XrmInitialize();

    /* Look for resource databases on server. */
    resource_string = XScreenResourceString( xscreen[s] );
    if ( resource_string != (char*) 0 )
	{
	rdb[s] = XrmGetStringDatabase( resource_string );
	XFree( resource_string );
	}
    else
	{
	resource_string = XResourceManagerString( displays[d_of_s[s]] );
	if ( resource_string != (char*) 0 )
	    rdb[s] = XrmGetStringDatabase( resource_string );
	else
	    {
	    /* No server databases, try ~/.Xdefaults */
	    char* cp;
	    char buf[500];

	    cp = getenv( "HOME" );
	    if ( cp != (char*) 0 )
		(void) strcpy( buf, cp );
	    else
		{
		struct passwd* pw;

		cp = getenv( "USER" );
		if ( cp != (char*) 0 )
		    pw = getpwnam( cp );
		else
		    pw = getpwuid( getuid() );
		if ( pw != (struct passwd*) 0 )
		    (void) strcpy( buf, pw->pw_dir );
		else
		    (void) strcpy( buf, "." );	/* best we can do */
		}
	    (void) strcat( buf, "/.Xdefaults" );
	    rdb[s] = XrmGetFileDatabase( buf );
	    }
	}

    /* Merge in XENVIRONMENT, if any. */
    xenv = getenv( "XENVIRONMENT" );
    if ( xenv != (char*) 0 )
	{
	xenv_rdb = XrmGetFileDatabase( xenv );
	XrmMergeDatabases( xenv_rdb, &rdb[s] );
	}

    /* And add command line options, but only for screen 0.  This is
    ** because XrmParseCommand removes the args from the command line,
    ** XrmMergeDatabases destroys the source database, and there is
    ** no routine to copy a database.
    */
    if ( s == 0 )
	XrmParseCommand(
	    &rdb[s], x_options, sizeof(x_options) / sizeof(*x_options),
	    app_name, argcP, argv );
    }

static char*
x_get_resource( s, name, class )
    int s;
    char* name;
    char* class;
    {
    char rname[500], rclass[500];
    char* type;
    XrmValue value;

    (void) sprintf( rname, "%s.%s", app_name, name );
    (void) sprintf( rclass, "%s.%s", X_CLASS, class );
    if ( XrmGetResource( rdb[s], rname, rclass, &type, &value ) == True )
	if ( strcmp( type, XtRString ) == 0 )
	    return (char*) value.addr;
    return (char*) 0;
    }

static int
x_str_to_bool( str )
    char* str;
    {
    if ( strcmp( str, "True" ) == 0 ||
         strcmp( str, "true" ) == 0 ||
         strcmp( str, "Yes" ) == 0 ||
         strcmp( str, "yes" ) == 0 ||
         strcmp( str, "On" ) == 0 ||
         strcmp( str, "on" ) == 0 ||
         strcmp( str, "Si" ) == 0 ||
         strcmp( str, "si" ) == 0 ||
         strcmp( str, "Da" ) == 0 ||
         strcmp( str, "da" ) == 0 ||
         strcmp( str, "T" ) == 0 ||
         strcmp( str, "t" ) == 0 ||
         strcmp( str, "Y" ) == 0 ||
         strcmp( str, "y" ) == 0 ||
         strcmp( str, "1" ) == 0 )
	return True;
    return False;
    }

static int
x_get_color_resource( s, name, class, cP )
    int s;
    char* name;
    char* class;
    unsigned long* cP;
    {
    char* rval;
    XColor color;

    rval = x_get_resource( s, name, class );
    if ( rval == (char*) 0 )
	return 0;
    if ( XParseColor(
	     displays[d_of_s[s]], DefaultColormapOfScreen( xscreen[s] ),
	     rval, &color ) != True )
	{
	(void) fprintf( stderr, "%s: can't parse color \"%s\"\n", argv0, rval );
	exit( 1 );
	}
    if ( XAllocColor(
	     displays[d_of_s[s]], DefaultColormapOfScreen( xscreen[s] ),
	     &color ) != True )
	{
	(void) fprintf(
	    stderr, "%s: can't allocate color \"%s\"\n", argv0, rval );
	exit( 1 );
	}
    *cP = color.pixel;
    return 1;
    }

static void
x_cleanup()
    {
    int d, s;

    for ( s = 0; s < nscreens; ++s )
	XFreeGC( displays[d_of_s[s]], rectgc[s] );
    for ( d = 0; d < ndisplays; ++d )
	XCloseDisplay( displays[d] );
    }

static void
reverse_screens()
    {
    int w, s;

    w = 0;
    for ( s = nscreens - 1; s >= 0; --s )
	{
	rootrect[s].x = w;
	w += rootrect[s].width;
	}
    }

static void
stealth()
    {
    int pid, tty;

    pid = fork();
    if ( pid < 0 )
	{
	perror( "fork" );
	exit( 1 );
	}
    else if ( pid > 0 )
	/* Parent just exits. */
	exit( 0 );
    (void) printf( "%d\n", getpid() );
    (void) fflush( stdout );

    /* Go stealth (ditch our controlling tty). */
    tty = open( "/dev/tty", 0 );
    if ( tty < 0 )
	{
	/* ENXIO means that there is no controlling terminal, so we don't
	** have to detach anything.
	*/
        if ( errno != ENXIO )
	    {
	    (void) fprintf( stderr, "%s: ", argv0 );
	    perror( "/dev/tty open" );
	    exit( 1 );
	    }
	}
    else
	{
	if ( ioctl( tty, TIOCNOTTY, 0 ) < 0 )
	    {
	    (void) fprintf( stderr, "%s: ", argv0 );
	    perror( "TIOCNOTTY ioctl" );
	    exit( 1 );
	    }
	(void) close( tty );
	}
    }

static void
rects_place()
    {
    int i, j, x0, y0, mh;

    /* Place bitmaps deterministically. */
    x0 = y0 = mh = 0;
    for ( i = 0; i < nrects; ++i )
	{
	if ( x0 + rects[i].bb.width > allrect.width )
	    {
	    y0 += mh;
	    x0 = mh = 0;
	    if ( y0 + rects[i].bb.height > allrect.height )
		{
		(void) fprintf( stderr, "%s: can't place bitmaps\n", argv0 );
		exit( 1 );
		}
	    }
	rects[i].bb.x = x0;
	rects[i].bb.y = y0;
	x0 += rects[i].bb.width;
	if ( rects[i].bb.height > mh )
	    mh = rects[i].bb.height;
	}
    /* Randomize positions. */
    for ( i = 0; i < INITIAL_JUMPS; ++i )
	for ( j = 0; j < nrects; ++j )
	    rect_jump( j );
    /* And initialize dx/dy/clock. */
    for ( i = 0; i < nrects; ++i )
	if ( rects[i].mode == M_JUMP )
	    rect_sleep( i );
	else if ( rects[i].mode == M_GLIDE || rects[i].mode == M_GRAVITY )
	    rect_bounce( i );
    }

static void
rect_bounce( i )
    int i;
    {
    do
	{
	rects[i].dx = random() % 9 - 4;
	rects[i].dy = random() % 9 - 4;
	}
    while ( ( rects[i].dx == 0 && rects[i].dy == 0 ) ||
	    abs( rects[i].dx ) > rects[i].bb.width ||
	    abs( rects[i].dy ) > rects[i].bb.height );
    }

static void
rect_sleep( i )
    int i;
    {
    rects[i].clock = random() % ( 5 * CYCLES );
    }

static void
rect_glide( i )
    int i;
    {
    int oldx, oldy, j;

    oldx = rects[i].bb.x;
    oldy = rects[i].bb.y;
    for ( j = 0; j < COLLISION_TRYS; ++j )
	{
	rects[i].bb.x += rects[i].dx;
	rects[i].bb.y += rects[i].dy;
	if ( ! check_collisions( i ) )
	    return;
	rects[i].bb.x = oldx;
	rects[i].bb.y = oldy;
	rect_bounce( i );
	}
    }

static void
rect_jump( i )
    int i;
    {
    int oldx, oldy, j;

    oldx = rects[i].bb.x;
    oldy = rects[i].bb.y;
    for ( j = 0; j < COLLISION_TRYS; ++j )
	{
	rects[i].bb.x = random() % ( allrect.width - rects[i].bb.width + 1 );
	rects[i].bb.y = random() % ( allrect.height - rects[i].bb.height + 1);
	if ( ! check_collisions( i ) )
	    return;
	}
    rects[i].bb.x = oldx;
    rects[i].bb.y = oldy;
    }

static void
main_loop()
    {
    FILE* cfP;
    int s, d, fd, maxfd;
    fd_set fds, fds2;
    struct timeval timeout, timeout2;
    XRectangle erect;

    /* Set up for signal catching. */
    goflag = 1;
    (void) signal( SIGHUP, sigcatch );
    (void) signal( SIGINT, sigcatch );
    (void) signal( SIGTERM, sigcatch );

    for ( s = 0; s < nscreens; ++s )
	XSelectInput( displays[d_of_s[s]], root[s], ExposureMask );

    if ( cycles != 0 )
	{
	FD_ZERO( &fds );
	maxfd = -1;
	for ( d = 0; d < ndisplays; ++d )
	    {
	    fd = ConnectionNumber( displays[d] );
	    FD_SET( fd, &fds );
	    maxfd = max( fd, maxfd );
	    }
	timeout.tv_usec = 1000000L / cycles;
	timeout.tv_sec = timeout.tv_usec / 1000000L;
	timeout.tv_usec = timeout.tv_usec % 1000000L;
	RectsInvalidate( &allrect, False );
	}

    gravclock = GRAVCONST;

    while ( goflag )
	{
	/* Check for invalid rectangles. */
	if ( RectsNextInvalid( &erect ) )
	    {
	    expose( &erect, False );
	    continue;
	    }
	/* Check for X events. */
	if ( check_events() )
	    continue;
	if ( cycles != 0 )
	    {
	    /* Nothing to do, so wait for a while. */
	    fds2 = fds;
	    timeout2 = timeout;
	    (void) select( maxfd + 1, &fds2, (int*) 0, (int*) 0, &timeout );
	    /* Check again for X events. */
	    if ( check_events() )
		continue;
	    }
	/* Nothing to do and we paused, so let's move bits. */
	move_rects();
	}
    
    for ( s = 0; s < nscreens; ++s )
	XClearArea(
	    displays[d_of_s[s]], root[s], 0, 0,
	    rootrect[s].width, rootrect[s].height, True );
    x_cleanup();
    exit( 0 );
    }

static void
sigcatch()
    {
    goflag = 0;
    }

static int
check_events()
    {
    int d;

    for ( d = 0; d < ndisplays; ++d )
	if ( XPending( displays[d] ) != 0 )
	    {
	    handle_event( d );
	    return 1;
	    }
    return 0;
    }

static void
handle_event( d )
    int d;
    {
    XEvent ev;
    int s;
    XRectangle erect;

    XNextEvent( displays[d], &ev );
    /*!!!*/
    switch ( ev.type )
	{
	case Expose:
	s = screen_of_window( ev.xexpose.window );
	if ( s != -1 )
	    {
	    erect.x = rootrect[s].x + ev.xexpose.x;
	    erect.y = ev.xexpose.y;
	    erect.width = ev.xexpose.width;
	    erect.height = ev.xexpose.height;
	    expose( &erect, True );
	    }
	break;
	}
    }

static int
screen_of_window( w )
    Window w;
    {
    int s;

    for ( s = 0; s < nscreens; ++s )
	if ( root[s] == w )
	    return s;
    return -1;
    }

static void
expose( erectP, real )
    XRectangle* erectP;
    Bool real;
    {
    int i, s;
    XRectangle irect;

    /* If the exposed area is totally in a rect, just paint it. */
    for ( i = 0; i < nrects; ++i )
	{
	if ( RectsInside( erectP, &rects[i].bb ) )
	    {
	    paint_rect( i, erectP );
	    return;
	    }
	}

    /* Paint the backgrounds if needed, then the exposed parts of any rects. */
    if ( ! real )
	for ( s = 0; s < nscreens; ++s )
	    if ( RectsTouch( erectP, &rootrect[s] ) )
		XClearArea(
		    displays[d_of_s[s]], root[s],
		    erectP->x - rootrect[s].x, erectP->y,
		    erectP->width, erectP->height, False );
    for ( i = 0; i < nrects; ++i )
	{
	irect = *erectP;
	RectsIntersect( &irect, &rects[i].bb );
	if ( irect.width > 0 && irect.height > 0 )
	    paint_rect( i, &irect );
	}
    }

static void
paint_rect( i, rectP )
    int i;
    XRectangle* rectP;
    {
    int s;

    for ( s = 0; s < nscreens; ++s )
	if ( RectsTouch( rectP, &rootrect[s] ) )
	    XCopyArea(
		displays[d_of_s[s]], rects[i].pixmap[s], root[s], rectgc[s],
		rectP->x - rects[i].bb.x, rectP->y - rects[i].bb.y,
		rectP->width, rectP->height,
		rectP->x - rootrect[s].x, rectP->y );
    }

static void
move_rects()
    {
    int i, oldx, oldy;
    XRectangle erect;

    --gravclock;
    for ( i = 0; i < nrects; ++i )
	if ( rects[i].mode == M_GLIDE || rects[i].mode == M_GRAVITY )
	    {
	    if ( rects[i].mode == M_GRAVITY && gravclock == 0 )
		++rects[i].dy;
	    oldx = rects[i].bb.x;
	    oldy = rects[i].bb.y;
	    rect_glide( i );

	    /* Erase old trailing edges. */
	    if ( rects[i].dx > 0 )
		{
		erect.x = oldx;
		erect.y = oldy;
		erect.width = rects[i].dx;
		erect.height = rects[i].bb.height;
		RectsInvalidate( &erect, False );
		}
	    else if ( rects[i].dx < 0 )
		{
		erect.x = rects[i].bb.x + rects[i].bb.width;
		erect.y = oldy;
		erect.width = -rects[i].dx;
		erect.height = rects[i].bb.height;
		RectsInvalidate( &erect, False );
		}
	    if ( rects[i].dy > 0 )
		{
		erect.x = oldx;
		erect.y = oldy;
		erect.width = rects[i].bb.width;
		erect.height = rects[i].dy;
		RectsInvalidate( &erect, False );
		}
	    else if ( rects[i].dy < 0 )
		{
		erect.x = oldx;
		erect.y = rects[i].bb.y + rects[i].bb.height;
		erect.width = rects[i].bb.width;
		erect.height = -rects[i].dy;
		RectsInvalidate( &erect, False );
		}

	    /* Paint whole bitmap in new position. */
	    RectsInvalidate( &rects[i].bb, False );
	    }
	else if ( rects[i].mode == M_JUMP )
	    {
	    --rects[i].clock;
	    if ( rects[i].clock <= 0 )
		{
		/* Erase whole bitmap in old position. */
		RectsInvalidate( &rects[i].bb, True );
		rect_jump( i );
		/* Paint whole bitmap in new position. */
		RectsInvalidate( &rects[i].bb, True );
		rect_sleep( i );
		}
	    }
    if ( gravclock <= 0 )
	gravclock = GRAVCONST;
    }

static int
check_collisions( i )
    int i;
    {
    int s, j;

    /* Check against boundary. */
    if ( ! RectsInside( &rects[i].bb, &allrect ) )
	return 1;

    /* Check against any smaller screens. */
    if ( smaller_screens )
	for ( s = 0; s < nscreens; ++s )
	    if ( rootrect[s].height < allrect.height )
		if ( rects[i].bb.x + rects[i].bb.width > rootrect[s].x &&
		     rects[i].bb.x < rootrect[s].x + rootrect[s].width &&
		     rects[i].bb.y + rects[i].bb.height > rootrect[s].height )
		    return 1;

    /* Check intersection with other rects. */
    for ( j = 0; j < nrects; ++j )
	if ( i != j )
	    if ( RectsTouch( &rects[i].bb, &rects[j].bb ) )
		return 1;

    return 0;
    }
