/* rects - rectangle routines
**
** Copyright (C) 1992 by Jef Poskanzer <jef@acme.com>.  All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <X11/Xlib.h>
#include <stdio.h>
#include "Rects.h"


/* Definitions. */

static int num_exposerects, max_exposerects;
static XRectangle* exposerects;


/* Routines. */

/* Initialize the Rects package. */
void
RectsInit()
    {
    num_exposerects = max_exposerects = 0;
    }

/* Queue up a rectangle for repainting - better than sending an Expose event. */
void
RectsInvalidate( rectP, merge )
    XRectangle* rectP;
    Bool merge;
    {
    int i;

    if ( merge )
	{
	/* Check if this rectangle intersects an existing one. */
	for ( i = 0; i < num_exposerects; ++i )
	    {
	    if ( RectsTouch( &exposerects[i], rectP ) )
		{
		RectsUnion( &exposerects[i], rectP );
		return;
		}
	    }
	}

    /* Nope, add a new XRectangle. */
    if ( num_exposerects == max_exposerects )
	{
	if ( max_exposerects == 0 )
	    {
	    max_exposerects = 32;	/* non-critical parameter */
	    exposerects = (XRectangle*) malloc(
		(unsigned) ( max_exposerects * sizeof(XRectangle) ) );
	    }
	else
	    {
	    max_exposerects *= 2;
	    exposerects = (XRectangle*) realloc(
		(char*) exposerects,
		(unsigned) ( max_exposerects * sizeof(XRectangle) ) );
	    }
	if ( exposerects == (XRectangle*) 0 )
	    {
	    (void) fprintf( stderr, "RectsInvalidate: out of memory\n" );
	    exit( 1 );
	    }
	}
    exposerects[num_exposerects] = *rectP;
    ++num_exposerects;
    }

/* Return a rectangle to be repainted, if there are any. */
Bool
RectsNextInvalid( rectP )
    XRectangle* rectP;
    {
    if ( num_exposerects == 0 )
	return False;
    --num_exposerects;
    *rectP = exposerects[num_exposerects];
    return True;
    }

/* Make rect1 the union of rect1 and rect2. */
void
RectsUnion( rect1P, rect2P )
    XRectangle* rect1P;
    XRectangle* rect2P;
    {
    if ( rect2P->x + rect2P->width > rect1P->x + rect1P->width )
	rect1P->width = rect2P->x + rect2P->width - rect1P->x;
    if ( rect2P->y + rect2P->height > rect1P->y + rect1P->height )
	rect1P->height = rect2P->y + rect2P->height - rect1P->y;
    if ( rect2P->x < rect1P->x )
	{
	rect1P->width = rect1P->x + rect1P->width - rect2P->x;
	rect1P->x = rect2P->x;
	}
    if ( rect2P->y < rect1P->y )
	{
	rect1P->height = rect1P->y + rect1P->height - rect2P->y;
	rect1P->y = rect2P->y;
	}
    }

/* Make rect1 the intersection of rect1 and rect2. */
void
RectsIntersect( rect1P, rect2P )
    XRectangle* rect1P;
    XRectangle* rect2P;
    {
    if ( ! RectsTouch( rect1P, rect2P ) )
	{
	rect1P->width = rect1P->height = 0;
	return;
	}
    if ( rect2P->x + rect2P->width < rect1P->x + rect1P->width )
	rect1P->width = rect2P->x + rect2P->width - rect1P->x;
    if ( rect2P->y + rect2P->height < rect1P->y + rect1P->height )
	rect1P->height = rect2P->y + rect2P->height - rect1P->y;
    if ( rect2P->x > rect1P->x )
	{
	rect1P->width = rect1P->x + rect1P->width - rect2P->x;
	rect1P->x = rect2P->x;
	}
    if ( rect2P->y > rect1P->y )
	{
	rect1P->height = rect1P->y + rect1P->height - rect2P->y;
	rect1P->y = rect2P->y;
	}
    }

/* Test whether rect1 and rect2 intersect. */
Bool
RectsTouch( rect1P, rect2P )
    XRectangle* rect1P;
    XRectangle* rect2P;
    {
    if ( rect1P->x + rect1P->width <= rect2P->x ||
         rect1P->y + rect1P->height <= rect2P->y ||
         rect2P->x + rect2P->width <= rect1P->x ||
         rect2P->y + rect2P->height <= rect1P->y )
	return False;
    return True;
    }

/* Test whether rect1 is inside rect2. */
Bool
RectsInside( rect1P, rect2P )
    XRectangle* rect1P;
    XRectangle* rect2P;
    {
    if ( rect1P->x >= rect2P->x &&
	 rect1P->x + rect1P->width <= rect2P->x + rect2P->width &&
         rect1P->y >= rect2P->y &&
	 rect1P->y + rect1P->height <= rect2P->y + rect2P->height )
	return True;
    return False;
    }
