/* sfcmilter - sender-forgery-checker mail filter module
**
** Copyright  2004 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**	notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**	notice, this list of conditions and the following disclaimer in the
**	documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/

#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_GRP_H
#include <grp.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#include <libmilter/mfapi.h>

#include "version.h"


/* Defines. */

#define HEADER_NAME "X-Sender-Check"



/* Structs. */


/* Per-connection data structure. */
#define DEL_HEADER_MAX 10
typedef struct {
    const char* ip_str;
    const char* helo;
    const char* mail_from;
    const char* envelope_from;
    const char* from;
    const char* sender;
    const char* resent_from;
    const char* resent_sender;
    int n_from, n_sender, n_resent_from, n_resent_sender;
    int n_my_headers;
    int n_del_headers;
    int del_header_list[DEL_HEADER_MAX];
    char buf[1000];
    } connection_data_t;

/* A word about buf, above.  This is a scratch buffer for temporary use
** by any routine that needs them.  Normally we'd just declare it locally,
** on the stack.  The problem is that we are running in a thread, each
** thread has a separate stack of fixed size, we don't know what that
** size is, and if we exceed it we dump core.  So we really want to
** avoid using up the stack with large local variable frames; instead,
** we use the per-connection data object which is allocated from the heap.
** It makes the connection objects fairly large, but not unreasonably so.
**
** We do have to be a little careful using the buffer, so that two
** routines in the same call stack don't try to make use of it simultaneously.
*/


/* Forwards. */

static void usage( void );
static void init_uid( const char* user );
static void init_socket( const char* sockpath );
static void fini_socket( const char* sockpath );
static void unlink_socket( const char* sockpath );
static void init_daemon( void );
static void init_pidfile( const char* pidfile );
static void fini_pidfile( const char* pidfile );

/* The milter calback routines.  Signatures for these are fixed
** by the milter API.
*/
static sfsistat sender_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr );
static sfsistat sender_helo( SMFICTX* ctx, char* helohost );
static sfsistat sender_envfrom( SMFICTX* ctx, char** fromargs );
static sfsistat sender_header( SMFICTX* ctx, char* name, char* value );
static sfsistat sender_eom( SMFICTX* ctx );
static sfsistat sender_abort( SMFICTX* ctx );
static sfsistat sender_close( SMFICTX* ctx );

static char* trim_smtp_address( char* addr );
static char* trim_header_address( char* addr );
static char* trim_whitespace( char* addr );
static int receiver_is_me( const char* value );
static int looks_forged( connection_data_t* cd );
static int strtailcasecmp( const char* s1, const char* s2 );
static void build_header( connection_data_t* cd, char* header, size_t header_size );
static connection_data_t* init_connection_data( void );
static int init_message_data( connection_data_t* cd );
static void fini_message_data( connection_data_t* cd );
static void fini_connection_data( connection_data_t* cd );


/* Globals. */

static char* argv0;
static int strict;

static char* local_hostname;
static int local_hostname_len;

static int header_name_len;

static struct smfiDesc smfilter = {
    "SENDER",				/* filter name */
    SMFI_VERSION,			/* version code -- do not change */
    SMFIF_CHGHDRS|SMFIF_ADDHDRS,	/* flags */
    sender_connect,			/* connection info filter */
    sender_helo,			/* SMTP HELO command filter */
    sender_envfrom,			/* envelope sender filter */
    NULL,				/* envelope recipient filter */
    sender_header,			/* header filter */
    NULL,				/* end of header */
    NULL,				/* body block filter */
    sender_eom,				/* end of message */
    sender_abort,			/* message aborted */
    sender_close			/* connection cleanup */
    };


int
main( int argc, char** argv )
    {
    int argn;
    char* user;
    char* pidfile;
    int nodaemon;
    char* sockpath;

    strict = 0;
    user = (char*) 0;
    pidfile = (char*) 0;
    nodaemon = 0;

    /* Figure out the program's name. */
    argv0 = strrchr( argv[0], '/' );
    if ( argv0 != (char*) 0 )
	++argv0;
    else
	argv0 = argv[0];

    header_name_len = strlen( HEADER_NAME );

    /* Parse args. */
    argn = 1;
    while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
	{
	if ( strncmp( argv[argn], "-strict", strlen( argv[argn] ) ) == 0 )
	    strict = 1;
	else if ( strncmp( argv[argn], "-user", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    user = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-pidfile", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    pidfile = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-nodaemon", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    nodaemon = 1;
	else if ( strncmp( argv[argn], "-X", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    nodaemon = 1;
	else
	    usage();
	++argn;
	}
    if ( argn >= argc )
	usage();
    sockpath = argv[argn++];
    if ( argn != argc )
	usage();

    local_hostname = (char*) 0;
    init_uid( user );
    init_socket( sockpath );
    openlog( argv0, 0, LOG_MAIL );
    if ( ! nodaemon )
	init_daemon();
    init_pidfile( pidfile );

    syslog( LOG_NOTICE, "%s %s starting", SFCMILTER_PROGRAM, SFCMILTER_VERSION );
    if ( smfi_main() == MI_FAILURE )
	{
	syslog( LOG_ERR, "smfi_main() failed" );
	exit( 1 );
	}
    syslog( LOG_NOTICE, "%s %s terminating", SFCMILTER_PROGRAM, SFCMILTER_VERSION );

    fini_pidfile( pidfile );
    fini_socket( sockpath );
    if ( local_hostname != (char*) 0 )
	free( (void*) local_hostname );

    closelog();

    return 0;
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage:  %s [-strict] [-user user] [-pidfile filename] [-nodaemon|-X] socket\n", argv0 );
    exit( 1 );
    }


static void
init_uid( const char* user )
    {
    struct passwd* pwd;
    char* ep;
    int uid;

    if ( getuid() == 0 )
	{
	/* If we're root, the very first thing we do is become another user. */
	if ( user == (char*) 0 )
	    (void) fprintf( stderr, "%s: warning: started as root but no --user flag specified\n", argv0 );
	else
	    {
	    /* Is it a number? */
	    uid = strtol( user, &ep, 0 );
	    if ( *ep == '\0' )
		pwd = getpwuid( uid );
	    else
		pwd = getpwnam( user );
	    if ( pwd == (struct passwd*) 0 )
		{
		(void) fprintf( stderr, "%s: unknown user: '%s'\n", argv0, user );
		exit( 1 );
		}
	    /* Set aux groups to null. */
	    if ( setgroups( 0, (gid_t*) 0 ) < 0 )
		{
		perror( "setgroups" );
		exit( 1 );
		}
	    /* Set primary group. */
	    if ( setgid( pwd->pw_gid ) < 0 )
		{
		perror( "setgid" );
		exit( 1 );
		}

	    /* Try setting aux groups correctly - not critical if this fails. */
	    if ( initgroups( user, pwd->pw_gid ) < 0 )
		    perror( "initgroups" );
	    /* Set uid. */
	    if ( setuid( pwd->pw_uid ) < 0 )
		{
		perror( "setuid" );
		exit( 1 );
		}
	    }
	}
    else
	{
	/* If we're not root but got a -user flag anyway, that's an error. */
	if ( user != (char*) 0 )
	    {
	    (void) fprintf( stderr, "%s: can't switch users if not started as root\n", argv0 );
	    exit( 1 );
	    }
	}
    }


static void
init_socket( const char* sockpath )
    {
    unlink_socket( sockpath );

    /* Harden our umask so that the new socket gets created securely. */
    umask( 0077 );

    /* Initialize milter stuff. */
    smfi_setconn( (char*) sockpath );
    if ( smfi_register( smfilter ) == MI_FAILURE )
	{
	(void) fprintf( stderr, "%s: smfi_register() failed\n", argv0 );
	exit( 1 );
	}
    }


static void
fini_socket( const char* sockpath )
    {
    unlink_socket( sockpath );
    }


static void
unlink_socket( const char* sockpath )
    {
    /* Remove old local socket.  Actually we should probably stat
    ** the file and check that it is in fact a socket before removing
    ** it.
    */
    if ( strncasecmp( sockpath, "unix:", 5 ) == 0 )
	{
	if ( unlink( &sockpath[5] ) < 0 )
	    if ( errno != ENOENT )
		perror( &sockpath[5] );
	}
    else if ( strncasecmp( sockpath, "local:", 6 ) == 0 )
	{
	if ( unlink( &sockpath[6] ) < 0 )
	    if ( errno != ENOENT )
		perror( &sockpath[6] );
	}
    else if ( strchr( sockpath, ':' ) == (char*) 0 )
	{
	if ( unlink( sockpath ) < 0 )
	    if ( errno != ENOENT )
		perror( sockpath );
	}
    }


static void
init_daemon( void )
    {
    /* Daemonize. */
#ifdef HAVE_DAEMON
    if ( daemon( 0, 0 ) < 0)
	{
	perror( "daemon" );
	exit( 1 );
	}
#else /* HAVE_DAEMON */
    switch ( fork() )
	{
	case 0:
	    break;  
	case -1:
	    syslog( LOG_CRIT, "fork: %m" );
	    perror( "fork" );
	    exit( 1 );
	default:
	    exit( 0 );
	}
#ifdef HAVE_SETSID
    setsid();
#endif /* HAVE_SETSID */
#endif /* HAVE_DAEMON */
    }


static void
init_pidfile( const char* pidfile )
    {
    if ( pidfile != (char*) 0 )
	{
	FILE* fp;

	fp = fopen( pidfile, "w" );
	if ( fp == (FILE*) 0 )
	    syslog( LOG_ERR, "unable to write PID file - %m" );
	else
	    {
	    (void) fprintf( fp, "%ld\n", (long) getpid() );
	    (void) fclose( fp );
	    }
	}
    }


static void
fini_pidfile( const char* pidfile )
    {
    if ( pidfile != (char*) 0 )
	(void) unlink( pidfile );
    }


/* Milter routines. */


/* sender_connect - handle the initial TCP connection
**
** Called at the start of a connection.  Any per-connection data should
** be initialized here.
**
** connhost: The hostname of the client, based on a reverse lookup.
** connaddr: The client's IP address, based on getpeername().
*/
static sfsistat
sender_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr )
    {
    connection_data_t* cd;
    struct sockaddr_in* sin;
    struct sockaddr_in6* sin6;
    const char* r;

    if ( connaddr == (_SOCK_ADDR*) 0 )
	return SMFIS_ACCEPT;	/* can't deal with it */

    /* Create & initialize the connection_data object. */
    cd = init_connection_data();
    if ( cd == (connection_data_t*) 0 )
	{
	syslog( LOG_ERR, "couldn't allocate connection_data" );
	return SMFIS_TEMPFAIL;
	}
    if ( smfi_setpriv( ctx, (void*) cd ) != MI_SUCCESS )
	{
	syslog( LOG_ERR, "smfi_setpriv() failed" );
	fini_connection_data( cd );
	return SMFIS_TEMPFAIL;
	}

    /* The first time through, we figure out the local hostname. */
    if ( local_hostname == (char*) 0 )
	{
	char* jmac;
	char lh[200];

	jmac = smfi_getsymval( ctx, "{j}" );
	if ( jmac != (char*) 0 )
	    local_hostname = strdup( jmac );
	else
	    {
	    if ( gethostname( lh, sizeof(lh) ) == 0 )
		local_hostname = strdup( lh );
	    else
		{
		syslog( LOG_ERR, "couldn't get local hostname" );
		return SMFIS_TEMPFAIL;
		}
	    }
	if ( local_hostname == (char*) 0 )
	    {
	    syslog( LOG_ERR, "couldn't strdup local hostname" );
	    return SMFIS_TEMPFAIL;
	    }
	local_hostname_len = strlen( local_hostname );
	}

    /* Figure out the IP address. */
    switch ( connaddr->sa_family )
	{
	case AF_INET:
	    sin = (struct sockaddr_in*) connaddr;
	    r = inet_ntop( AF_INET, &sin->sin_addr, cd->buf, sizeof(cd->buf) );
	    if ( r == (char*) 0 )
		{
		syslog( LOG_ERR, "couldn't convert IPv4 address to a string - %m" );
		return SMFIS_TEMPFAIL;
		}
	    break;
	case AF_INET6:
	    sin6 = (struct sockaddr_in6*) connaddr;
	    r = inet_ntop( AF_INET6, &sin6->sin6_addr, cd->buf, sizeof(cd->buf) );
	    if ( r == (char*) 0 )
		{
		syslog( LOG_ERR, "couldn't convert IPv6 address to a string - %m" );
		return SMFIS_TEMPFAIL;
		}
	    break;
	default:
	    syslog( LOG_ERR, "unknown address family" );
	    return SMFIS_TEMPFAIL;
	}
    cd->ip_str = strdup( cd->buf );
    if ( cd->ip_str == (char*) 0 )
	{
	syslog( LOG_ERR, "couldn't strdup ip_str" );
	return SMFIS_TEMPFAIL;
	}

    return SMFIS_CONTINUE;
    }


/* sender_helo - handle the HELO command
**
** Called at the start of a connection.
**
** helohost: The string passed to the HELO/EHLO command.
*/
static sfsistat
sender_helo( SMFICTX* ctx, char* helohost )
    {
    connection_data_t* cd;

    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( cd->helo != (char*) 0 )
	syslog( LOG_NOTICE, "multiple HELOs from %s [%s] - ignoring '%s'", cd->helo, cd->ip_str, helohost );
    else
	{
	cd->helo = strdup( trim_whitespace( helohost ) );
	if ( cd->helo == (char*) 0 )
	    {
	    syslog( LOG_ERR, "couldn't strdup helo" );
	    return SMFIS_TEMPFAIL;
	    }
	}

    return SMFIS_CONTINUE;
    }


/* sender_envfrom - handle the MAIL FROM:<> command
**
** Called at the start of each message.  There may be multiple messages
** in a connection.  Any per-message data should be initialized here.
**
** fromargs: Null-terminated SMTP command arguments; fromargs[0] is the
**		   sender address.
*/
static sfsistat
sender_envfrom( SMFICTX* ctx, char** fromargs )
    {
    connection_data_t* cd;

    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( ! init_message_data( cd ) )
	{
	syslog( LOG_ERR, "init_message_data() failed" );
	return SMFIS_TEMPFAIL;
	}

    cd->mail_from = strdup( trim_smtp_address( fromargs[0] ) );
    if ( cd->mail_from == (char*) 0 )
	{
	syslog( LOG_ERR, "couldn't strdup mail_from" );
	smfi_setreply( ctx, "452", "4.3.1", "Please try again later - couldn't strdup mail_from." );
	fini_message_data( cd );
	return SMFIS_TEMPFAIL;
	}

    /* If the mail-from is empty or doesn't have an @, we make a fake one
    ** using the HELO domain (or if that's missing, the IP address).
    */
    if ( cd->mail_from[0] == '\0' || strchr( cd->mail_from, '@' ) == (char*) 0 )
	{
	if ( cd->helo != (char*) 0 )
	    (void) snprintf( cd->buf, sizeof(cd->buf), "mailer_daemon@%s", cd->helo );
	else
	    (void) snprintf( cd->buf, sizeof(cd->buf), "mailer_daemon@[%s]", cd->ip_str );
	cd->envelope_from = strdup( cd->buf );
	}
    else
	cd->envelope_from = strdup( cd->mail_from );
    if ( cd->envelope_from == (char*) 0 )
	{
	syslog( LOG_ERR, "couldn't strdup envelope_from" );
	smfi_setreply( ctx, "452", "4.3.1", "Please try again later - couldn't strdup envelope_from." );
	fini_message_data( cd );
	return SMFIS_TEMPFAIL;
	}

    return SMFIS_CONTINUE;
    }


/* sender_header - handle a header line
**
** Called separately for each header line in a message.
**
** name:  Header field name.
** value: Header vield value, including folded whitespace.  The final CRLF
**		is removed.
*/
static sfsistat
sender_header( SMFICTX* ctx, char* name, char* value )
    {
    connection_data_t* cd;
    
    cd = (connection_data_t*) smfi_getpriv( ctx );

    /* We look for any instances of our own header in the incoming
    ** message, so we can remove them later.
    */
    if ( strcasecmp( name, HEADER_NAME ) == 0 )
	{
	++cd->n_my_headers;

	/* Does this header claim to be from us? */
	if ( receiver_is_me( value ) )
	    {
	    syslog( LOG_NOTICE, "found possible spoofed header - '%s'", value );
	    if ( cd->n_del_headers < DEL_HEADER_MAX )
		{
		cd->del_header_list[cd->n_del_headers] = cd->n_my_headers;
		++cd->n_del_headers;
		}
	    }
	}

    /* We also copy the various sender headers. */
    else if ( strcasecmp( name, "From" ) == 0 )
	{
	if ( cd->from == (char*) 0 )
	    {
	    cd->from = strdup( trim_header_address( value ) );
	    if ( cd->from == (char*) 0 )
		{
		syslog( LOG_ERR, "couldn't strdup From" );
		return SMFIS_TEMPFAIL;
		}
	    }
	++cd->n_from;
	}
    else if ( strcasecmp( name, "Sender" ) == 0 )
	{
	if ( cd->sender == (char*) 0 )
	    {
	    cd->sender = strdup( trim_header_address( value ) );
	    if ( cd->sender == (char*) 0 )
		{
		syslog( LOG_ERR, "couldn't strdup Sender" );
		return SMFIS_TEMPFAIL;
		}
	    }
	++cd->n_sender;
	}
    else if ( strcasecmp( name, "Resent-From" ) == 0 )
	{
	if ( cd->resent_from == (char*) 0 )
	    {
	    cd->resent_from = strdup( trim_header_address( value ) );
	    if ( cd->resent_from == (char*) 0 )
		{
		syslog( LOG_ERR, "couldn't strdup Resent-From" );
		return SMFIS_TEMPFAIL;
		}
	    }
	++cd->n_resent_from;
	}
    else if ( strcasecmp( name, "Resent-Sender" ) == 0 )
	{
	if ( cd->resent_sender == (char*) 0 )
	    {
	    cd->resent_sender = strdup( trim_header_address( value ) );
	    if ( cd->resent_sender == (char*) 0 )
		{
		syslog( LOG_ERR, "couldn't strdup Resent-Sender" );
		return SMFIS_TEMPFAIL;
		}
	    }
	++cd->n_resent_sender;
	}

    return SMFIS_CONTINUE;
    }


/* sender_eom - handle the end of the message
**
** Called once per message after all body blocks have been processed.
** Any per-message data should be freed both here and in sender_abort(),
** and wherever you return any of SMFIS_REJECT/SMFIS_TEMPFAIL/SMFIS_DISCARD.
*/
static sfsistat
sender_eom( SMFICTX* ctx )
    {
    connection_data_t* cd;
    int i;

    cd = (connection_data_t*) smfi_getpriv( ctx );

    /* Header deleting and adding can only happen in the eom handler. */

    /* Remove any previous headers, to prevent spoofing. */
    if ( cd->n_del_headers >= DEL_HEADER_MAX )
	{
	/* There were too many bogus headers to keep track of, so to
	** be safe we just delete all of our headers.
	*/
	for ( i = cd->n_my_headers; i >= 1; --i )
	    smfi_chgheader( ctx, HEADER_NAME, i, (char*) 0 );
	}
    else
	{
	/* We have a list of only the headers purporting to be from us;
	** those get deleted.  We do the deletions in reverse order in case
	** deleting one header causes subsequent headers to get re-numbered.
	** (I actually checked whether this happens in the current milter
	** implementation, and it does NOT, but perhaps some future version
	** will get it wrong.)
	*/
	for ( i = cd->n_del_headers - 1 ; i >= 0; --i )
	    smfi_chgheader( ctx, HEADER_NAME, cd->del_header_list[i], (char*) 0 );
	}

    /* Is the header-sender forged? */
    if ( looks_forged( cd ) )
	{
	build_header( cd, cd->buf, sizeof(cd->buf) );
	smfi_addheader( ctx, HEADER_NAME, cd->buf );
	}

    fini_message_data( cd );

    return SMFIS_CONTINUE;
    }


/* sender_abort - handle the message being aborted
**
** May be called at any time during processing of a message, indicating
** that something went wrong.  Any per-message data should be freed both
** here and in sample_eom(), and wherever you return any of
** SMFIS_REJECT/SMFIS_TEMPFAIL/SMFIS_DISCARD.
*/
static sfsistat
sender_abort( SMFICTX* ctx )
    {
    connection_data_t* cd;
    
    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( cd != (connection_data_t*) 0 )
	fini_message_data( cd );
    else
	syslog( LOG_NOTICE, "sender_abort called but cd is null" );

    return SMFIS_CONTINUE;
    }


/* sender_close - handle the connection being closed
**
** Called once at the end of a connection.  Any per-connection data
** should be freed here.
*/
static sfsistat
sender_close( SMFICTX* ctx )
    {
    connection_data_t* cd;

    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( cd != (connection_data_t*) 0 )
	{
	smfi_setpriv( ctx, (void*) 0 );
	fini_connection_data( cd );
	}
    else
	syslog( LOG_NOTICE, "sender_close called but cd is null" );

    return SMFIS_CONTINUE;
    }


static char*
trim_smtp_address( char* addr )
    {
    int len;

    /* The milter API sometimes gives us strings with angle brackets. */
    while ( *addr == '<' )
	++addr;
    len = strlen( addr );
    while ( len > 0 && addr[len - 1] == '>' )
	addr[--len] = '\0';

    /* Finally, trim any leading or trailing whitespace. */
    return trim_whitespace( addr );
    }


static char*
trim_header_address( char* addr )
    {
    char* left;
    char* right;

    /* First, remove any parenthesized comments. */
    while ( ( right = strchr( addr, ')' ) ) != (char*) 0 )
	{
	for ( left = right; ; --left )
	    {
	    if ( left == addr || *left == '(' )
		{
		(void) strcpy( left, right + 1 );
		break;
		}
	    }
	}

    /* If there are angle brackets then the real address is inside them. */
    left = strchr( addr, '<' );
    if ( left != (char*) 0 )
	{
	addr = left + 1;
	right = strchr( addr, '>' );
	if ( right != (char*) 0 )
	    *right = '\0';
	}

    /* Finally, trim any leading or trailing whitespace. */
    return trim_whitespace( addr );
    }


static char*
trim_whitespace( char* addr )
    {
    int len;

    len = strlen( addr );
    while ( len > 0 && ( addr[0] == ' ' || addr[0] == '\t' || addr[0] == '\012' || addr[0] == '\015' ) )
	{
	++addr;
	--len;
	}
    while ( len > 0 && ( addr[len-1] == ' ' || addr[len-1] == '\t' || addr[len-1] == '\012' || addr[len-1] == '\015' ) )
	{
	--len;
	addr[len] = '\0';
	}

    return addr;
    }


static int
receiver_is_me( const char* value )
    {
    const char* cp;

    /* Look for a receiver= setting.  The hostname immediately follows the
    ** equals and is immediately followed by a semicolon.
    */
    cp = strstr( value, " receiver=" );
    if ( cp != (char*) 0 )
	{
	cp += 10;
	if ( strncasecmp( cp, local_hostname, local_hostname_len ) == 0 )
	    {
	    cp += local_hostname_len;
	    if ( *cp == ';' )
		return 1;	/* Yes! */
	    }
	}

    /* Nope. */
    return 0;
    }


static int
looks_forged( connection_data_t* cd )
    {
    const char* message_from;
    const char* envelope_from_hostname;
    const char* message_from_hostname;

    /* If ther was no usable envelope-from, we can't check. */
    if ( cd->envelope_from == (char*) 0 )
	return 0;

    /* Figure out which string we should compare against the envelope-from. */
    message_from = (char*) 0;
    if ( cd->n_resent_sender == 1 )
	message_from = cd->resent_sender;
    else if ( cd->n_resent_from == 1 )
	message_from = cd->resent_from;
    else if ( cd->n_sender == 1 )
	message_from = cd->sender;
    else if ( cd->n_from == 1 )
	message_from = cd->from;

    if ( message_from != (char*) 0 )
	{
	if ( strict )
	    {
	    /* Strict comparison - just check the two entire strings. */
	    if ( strcasecmp( cd->envelope_from, message_from ) != 0 )
		return 1;
	    else
		return 0;
	    }
	else
	    {
	    /* Non-strict comparison.  This is more complicated. */

	    /* First, we only look at the hostname parts of the two strings,
	    ** because that's all that spfmilter looks at and we are aiming
	    ** to be a good companion to spfmilter.  Yes this allows forgeries
	    ** within the same hostname to pass through - joe@mail.acme.com
	    ** masquerading as jane@mail.acme.com.  If those are important to you,
	    ** then use the -strict flag.
	    */
	    envelope_from_hostname = strchr( cd->envelope_from, '@' );
	    if ( envelope_from_hostname == (char*) 0 )
		envelope_from_hostname = cd->envelope_from;
	    else
		++envelope_from_hostname;
	    message_from_hostname = strchr( message_from, '@' );
	    if ( message_from_hostname == (char*) 0 )
		message_from_hostname = message_from;
	    else
		++message_from_hostname;

	    /* Now do the actual comparison. */
	    if ( strtailcasecmp( envelope_from_hostname, message_from_hostname ) != 0 )
		return 1;
	    else
		return 0;
	    }
	}
    }


/* Compares either the tail of s1 to s2, or s1 to the tail of s2, whichever
** is shorter.
*/
static int
strtailcasecmp( const char* s1, const char* s2 )
    {
    int l1 = strlen( s1 );
    int l2 = strlen( s2 );

    if ( l1 >= l2 )
	return strcasecmp( &s1[l1-l2], s2 );
    else
	return strcasecmp( s1, &s2[l2-l1] );
    }


static void
build_header( connection_data_t* cd, char* header, size_t header_size )
    {
    int len;

    (void) snprintf( header, header_size, "possibly forged;" );
    len = strlen( header );
    if ( local_hostname != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " receiver=%s;", local_hostname );
	len = strlen( header );
	}
    if ( cd->ip_str!= (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " client-ip=%s;", cd->ip_str);
	len = strlen( header );
	}
    if ( cd->helo != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " helo=%s;", cd->helo );
	len = strlen( header );
	}
    if ( cd->mail_from != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " envelope-from=%s;", cd->mail_from );
	len = strlen( header );
	}
    if ( cd->from != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " from=%s;", cd->from );
	len = strlen( header );
	}
    if ( cd->sender != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " sender=%s;", cd->sender );
	len = strlen( header );
	}
    if ( cd->resent_from != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " resent_from=%s;", cd->resent_from );
	len = strlen( header );
	}
    if ( cd->resent_sender != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " resent_sender=%s;", cd->resent_sender );
	len = strlen( header );
	}
    (void) snprintf( &header[len], header_size - len, " software=%s %s %s;", SFCMILTER_PROGRAM, SFCMILTER_VERSION, SFCMILTER_URL );
    }


static connection_data_t*
init_connection_data( void )
    {
    connection_data_t* cd;

    cd = (connection_data_t*) malloc( sizeof(connection_data_t) );
    if ( cd == (connection_data_t*) 0 )
	return (connection_data_t*) 0;
    cd->ip_str = (char*) 0;
    cd->helo = (char*) 0;
    if ( ! init_message_data( cd ) )
	{
	fini_connection_data( cd );
	return (connection_data_t*) 0;
	}
    return cd;
    }


static int
init_message_data( connection_data_t* cd )
    {
    cd->mail_from = (char*) 0;
    cd->envelope_from = (char*) 0;
    cd->from = (char*) 0;
    cd->sender = (char*) 0;
    cd->resent_from = (char*) 0;
    cd->resent_sender = (char*) 0;
    cd->n_from = cd->n_sender = cd->n_resent_from = cd->n_resent_sender = 0;
    cd->n_my_headers = 0;
    cd->n_del_headers = 0;
    return 1;
    }


/* Frees or resets any items in the connection_data that are for
** an individual message.
*/
static void
fini_message_data( connection_data_t* cd )
    {
    if ( cd->mail_from != (char*) 0 )
	{
	free( (void*) cd->mail_from );
	cd->mail_from = (char*) 0;
	}
    if ( cd->envelope_from != (char*) 0 )
	{
	free( (void*) cd->envelope_from );
	cd->envelope_from = (char*) 0;
	}
    if ( cd->from != (char*) 0 )
	{
	free( (void*) cd->from );
	cd->from = (char*) 0;
	}
    if ( cd->sender != (char*) 0 )
	{
	free( (void*) cd->sender );
	cd->sender = (char*) 0;
	}
    if ( cd->resent_from != (char*) 0 )
	{
	free( (void*) cd->resent_from );
	cd->resent_from = (char*) 0;
	}
    if ( cd->resent_sender != (char*) 0 )
	{
	free( (void*) cd->resent_sender );
	cd->resent_sender = (char*) 0;
	}
    }


/* Frees all items in the connection_data. */
static void
fini_connection_data( connection_data_t* cd )
    {
    fini_message_data( cd );
    if ( cd->ip_str != (char*) 0 )
	free( (void*) cd->ip_str );
    if ( cd->helo != (char*) 0 )
	free( (void*) cd->helo );
    free( (void*) cd );
    }
