/* SecurePage - make self-decrypting web pages
**
** Copyright  2010,2013 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://acme.com/license.html
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "stripe.h"


static void arc4_init( unsigned char* key, unsigned int key_length );
static unsigned char arc4_output();
static void arc4_swap( unsigned char* s, unsigned int a, unsigned int b );

#define HASH_MUL 241
#define HASH_MOD 8388593
#define DISCARD 1024


int
main( int argc, char** argv )
    {
    char* argv0;
    int argn;
    char* title;
    char passphrase[1000];
    char* cp;
    char salt[50];
    int c;
    int hash;
    int first;
    char* usage = "usage:  %s [-t title] passphrase < clearpage > cypherpage\n";

    /* Check args. */
    argv0 = argv[0];
    title = (char*) 0;
    argn = 1;
    while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )  
	{
	if ( strcmp( argv[argn], "-t" ) == 0 )
	    {
	    ++argn;
	    title = argv[argn];
	    }
	else
	    {
	    (void) fprintf( stderr, usage, argv0 );
	    exit( 1 );
	    }
	++argn;
	}
    if ( argn != argc - 1 )
	{
	(void) fprintf( stderr, usage, argv0 );
	exit( 1 );
	}

    /* Copy and clear the passphrase. */
    (void) strncpy( passphrase, argv[argn], sizeof(passphrase) );
    for ( cp = argv[argn]; *cp != '\0'; ++cp )
	*cp = '\0';

    /* Make salt. */
    srandomdev();
    (void) snprintf( salt, sizeof(salt), "%08lx%08lx", random(), random() );
    (void) strlcat( passphrase, salt, sizeof(passphrase) - strlen( passphrase ) - 1 );

    /* Write out the preamble. */
    (void) printf( "\
<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">\n\
<html>\n\
  <head>\n\
    <meta http-equiv=\"Content-type\" content=\"text/html;charset=UTF-8\">\n\
    <title>SecurePage Encrypted</title>\n\
  </head>\n\
  <body style=\"margin-left: 60px; background-color: #f3f1c8; background-image: url('data:image/png;base64,%s');\">\n\
    <h3>SecurePage Encrypted</h3>\n\
", stripe_data );
    if ( title != (char*) 0 )
	(void) printf( "    <h4>%s</h4>\n", title );
    (void) printf( "\
    <p>\n\
    This page is passphrase-protected.\n\
    <p>\n\
    <form method=\"get\" action=\"javascript:void(0);\" onsubmit=\"Decrypt();\">\n\
      <input id=\"passphrase\" size=\"30\" type=\"password\">\n\
      <input type=\"submit\" value=\"Decrypt\">\n\
    </form>\n\
    <hr>\n\
    <a href=\"http://acme.com/software/securepage/\" target=\"_top\">SecurePage</a> &copy;2010 by <a href=\"http://acme.com/\" target=\"_top\">ACME Laboratories</a>\n\
    <script type=\"text/javascript\">\n\
//<![CDATA[\n\
// SecurePage decryption.\n\
// Copyright  2010 by Jef Poskanzer <jef@mail.acme.com>.\n\
var salt = '%s';\n\
var code = [", salt );

    /* Process the cleartext. */
    arc4_init( passphrase, strlen( passphrase ) );
    hash = 0;
    first = 1;
    for (;;)
	{
	c = getchar();
	if ( c == EOF )
	    break;
	hash = ( hash * HASH_MUL + c ) % HASH_MOD;
	c = ( c ^ arc4_output() ) & 255;
	if ( ! first )
	    putchar( ',' );
	(void) printf( "%d", c );
	first = 0;
	}

    /* Write out the epilog. */
    (void) printf( "\
];\n\
var hashMul = %d;\n\
var hashMod = %d;\n\
var discard = %d;\n\
var clearHash = %d;\n\
var passphraseElement = document.getElementById( 'passphrase' );\n\
\n\
function Decrypt()\n\
    {\n\
    var passphrase = passphraseElement.value + salt;\n\
    arc4_init( passphrase, passphrase.length );\n\
    var clear = [];\n\
    var hash = 0;\n\
    for ( var i = 0; i < code.length; ++i )\n\
	{\n\
	clear[i] = ( code[i] ^ arc4_output() ) & 255;\n\
	hash = ( hash * hashMul + clear[i] ) %% hashMod;\n\
	}\n\
    if ( hash != clearHash )\n\
	{\n\
	alert( \"Incorrect passphrase!\" );\n\
	passphraseElement.value = \"\";\n\
	return;\n\
	}\n\
    var clearText = \"\";\n\
    for ( var i = 0; i < clear.length; ++i )\n\
	clearText += String.fromCharCode( clear[i] );\n\
    // MSIE doesn't like us to modify the documentElement, so we use
    // the body instead.  This is not quite right but it works.
    var e = document.getElementsByTagName('body')[0];\n\
    ClearElement( e );\n\
    e.innerHTML = clearText;\n\
    }\n\
\n\
function ClearElement( element )\n\
    {\n\
    while ( element.firstChild != null )\n\
	{\n\
	ClearElement( element.firstChild );\n\
	element.removeChild( element.firstChild );\n\
	}\n\
    }\n\
\n\
// ARC4 stream cipher, JavaScript version.\n\
\n\
var arc4_S = new Array( 256 );
var arc4_i, arc4_j;
\n\
function arc4_init( key, key_length )\n\
    {\n\
    var a, b;\n\
    for ( a = 0; a < 256; ++a )\n\
        arc4_S[a] = a;\n\
    for ( a = b = 0; a < 256; ++a )\n\
	{\n\
        b = ( b + key.charCodeAt( a %% key_length ) + arc4_S[a] ) & 255;\n\
        arc4_swap( arc4_S, a, b );\n\
	}
    arc4_i = arc4_j = 0;\n\
    // Discard the first few elements.\n\
    for ( a = 0; a < discard; ++a )\n\
	arc4_output();\n\
    }\n\
\n\
function arc4_output()\n\
    {\n\
    arc4_i = ( arc4_i + 1 ) & 255;\n\
    arc4_j = ( arc4_j + arc4_S[arc4_i] ) & 255;\n\
    arc4_swap( arc4_S, arc4_i, arc4_j );\n\
    return arc4_S[( arc4_S[arc4_i] + arc4_S[arc4_j] ) & 255];\n\
    }\n\
\n\
function arc4_swap( s, a, b )\n\
    {\n\
    var temp = s[a];\n\
    s[a] = s[b];\n\
    s[b] = temp;\n\
    }\n\
//]]>\n\
    </script>\n\
  </body>\n\
</html>\n\
", HASH_MUL, HASH_MOD, DISCARD, hash );

    exit( 0 );
    }


/* ARC4 stream cipher, C version. */

static unsigned char arc4_S[256];
static unsigned int arc4_i, arc4_j;

static void
arc4_init( unsigned char* key, unsigned int key_length )
    {
    int a, b;
    for ( a = 0; a < 256; ++a )
        arc4_S[a] = a;
    for ( a = b = 0; a < 256; ++a )
	{
        b = ( b + key[a % key_length] + arc4_S[a] ) & 255;
        arc4_swap( arc4_S, a, b );
	}
    arc4_i = arc4_j = 0;
    /* Discard the first few elements. */
    for ( a = 0; a < DISCARD; ++a )
	(void) arc4_output();
    }

static unsigned char
arc4_output()
    {
    arc4_i = ( arc4_i + 1 ) & 255;
    arc4_j = ( arc4_j + arc4_S[arc4_i] ) & 255;
    arc4_swap( arc4_S, arc4_i, arc4_j );
    return arc4_S[( arc4_S[arc4_i] + arc4_S[arc4_j] ) & 255];
    }

static void
arc4_swap( unsigned char* s, unsigned int a, unsigned int b )
    {
    unsigned char temp = s[a];
    s[a] = s[b];
    s[b] = temp;
    }
