/* say - front-end for write(1) that records messages
**
** Copyright (C) 1990,1991 by Jef Poskanzer <jef@acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdio.h>
#include <string.h>
#include <sys/file.h>
#include <fcntl.h>
#include <sys/types.h>
#include <signal.h>
#include <utmp.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <pwd.h>
#include <ctype.h>
#include <errno.h>
#include "libsaywha.h"


/* Definitions. */

#ifndef _PATH_UTMP
#define _PATH_UTMP "/etc/utmp"
#endif

#ifndef UT_LINESIZE
#define UT_LINESIZE sizeof(((struct utmp *)0)->ut_line)
#endif


/* Externals. */

extern int dup2();
extern int fork();
extern int fprintf();
extern int isatty();
extern int printf();
extern int system();
extern int vfork();
extern char* ctime();
extern char* getenv();
extern char* getlogin();
extern char* mktemp();
extern char* ttyname();
extern time_t time();


/* Forward routines. */

static void cleanup();
static int search_utmp();
#define SU_OK 0
#define SU_NOTFOUND 1
#define SU_MSGSOFF 2


/* Private globals. */

static char tmpfilename[] = "/tmp/say.XXXXXX";
static int unlinktmpfile = 0;
char sayfilename[200];
int sayfilefd;
static int unlocksayfile = 0;
char mysayfilename[200];
int mysayfilefd;
static int unlockmysayfile = 0;


/* Routines. */

int
main( argc, argv )
    int argc;
    char* argv[];
    {
    struct stat termstat;
    char buf[1000];
    FILE* fp;
    int su, ist, child, wval;
    int waitstatus;
    int mail_only = 0;

    sw_argv0 = argv[0];

    /* Initialize. */
    (void) signal( SIGINT, cleanup );
    (void) signal( SIGHUP, cleanup );
    if ( sw_check_sayrc() < 0 )
	(void) cleanup( 1 );
    if ( sw_check_my_sayfile() < 0 )
	(void) cleanup( 1 );

    /* Cursory check of args - let write(1) do the rigorous check. */
    if ( argc != 2 && argc != 3 )
	{
	(void) fprintf( stderr, "usage:  %s <user> [ <tty> ]\n", sw_argv0 );
	(void) cleanup( 1 );
	}
    su = search_utmp( argv[1] );
    if ( su == SU_NOTFOUND )
	{
	if ( getpwnam( argv[1] ) != (struct passwd*) 0 )
	    {
	    (void) fprintf(
		stderr, "%s: %s is not logged in, will mail\n",
		sw_argv0, argv[1] );
	    mail_only = 1;
	    }
	else
	    {
	    (void) fprintf( stderr, "%s: no such user\n", sw_argv0 );
	    (void) cleanup( 1 );
	    }
	}
    else if ( su == SU_MSGSOFF )
	{
	(void) fprintf(
	    stderr, "%s: %s has messages disabled\n", sw_argv0, argv[1] );
	(void) cleanup( 1 );
	}
    if ( stat( ttyname( fileno( stderr ) ), &termstat ) >= 0 )
	{
	if ( ( termstat.st_mode & (S_IWRITE>>3) ) == 0 )  /* group write bit */
	    {
	    (void) fprintf( stderr,
		"%s: you have write permission turned off\n", sw_argv0 );
	    (void) cleanup( 1 );
	    }
	}

    /* Read in the message and save it to a temp file. */
    (void) mktemp( tmpfilename );
    if ( ( fp = fopen( tmpfilename, "w" ) ) == (FILE*) 0 )
	{
	(void) fprintf(
	    stderr, "%s: couldn't open temp file %s - %s\n", sw_argv0,
	    tmpfilename, strerror( errno ) );
	(void) cleanup( 1 );
	}
    unlinktmpfile = 1;
    (void) fchmod( fileno( fp ), 0600 );
#ifdef MAIL_EDITOR
    if ( sw_mail_editor )
	{
	(void) sprintf( buf, "%s %s", MAIL_EDITOR, tmpfilename );
	(void) system( buf );
	}
    else
	{
#endif
    if ( ( ist = isatty( fileno( stdin ) ) ) && ! sw_terse )
	(void) printf( "Enter message, '^D' or '.' to end.\n" );
    while ( fgets( buf, sizeof( buf ), stdin ) != (char*) 0 )
	{
	int i;

	if ( buf[0] == '.' && buf[1] == '\n' && buf[2] == '\0' && ist )
	    break;
	/* Write the line with control characters made visible. */
	for ( i = 0; buf[i] != '\0'; ++i )
	    {
	    char c;

	    c = toascii( buf[i] );
	    if ( ! isprint( c ) && ! isspace( c ) && c != '\n' && c != '\007' )
		{
		(void) putc( '^', fp );
		(void) putc( c ^ 0x40, fp );	/* DEL to ?, others to alpha */
		}
	    else
		(void) putc( c, fp );
	    }
	}
    if ( fclose( fp ) == EOF )
	{
	(void) fprintf(
	    stderr, "%s: error closing temp file %s - %s\n", sw_argv0,
	    tmpfilename, strerror( errno ) );
	(void) cleanup( 1 );
	}
#ifdef MAIL_EDITOR
	}
#endif

    /* Try sending it. */
    if ( mail_only )
	goto mail;
    if ( ( child = vfork() ) == 0 )
	{
	int fd;

	if ( ( fd = open( tmpfilename, O_RDONLY ) ) < 0 )
	    {
	    (void) fprintf(
		stderr, "%s: couldn't re-open temp file %s - %s\n", sw_argv0,
		tmpfilename, strerror( errno ) );
	    (void) cleanup( 1 );
	    }
	(void) dup2( fd, fileno( stdin ) );
	if ( argc == 2 )
	    (void) execlp( WRITE, WRITE, argv[1], (char*) 0 );
	else
	    (void) execlp( WRITE, WRITE, argv[1], argv[2], (char*) 0 );
	(void) fprintf(
	    stderr, "%s: exec failed - %s\n", sw_argv0, strerror( errno ) );
	exit( 1 );
	}
    while ( ( wval = wait( &waitstatus ) ) != child && wval != -1 )
	;
    if ( wval == child && WTERMSIG(waitstatus) == 0 &&
	 WEXITSTATUS(waitstatus) == 0 )
	{
	/* Success.  Now save a copy in the recipient's .sayfile, and the
	** sender's too if sw_savemine is set. */
	struct passwd* pw;
	char* home;
	char* login;
	char* nows;
	char host[100];
	time_t now;

#ifdef BACKGROUND
	if ( fork() > 0 )
	    exit( 0 );
#endif /*BACKGROUND*/

	if ( ( pw = getpwnam( argv[1] ) ) == (struct passwd*) 0 )
	    {
	    (void) fprintf( stderr, "%s: couldn't find home directory for %s\n",
		sw_argv0, argv[1] );
	    (void) cleanup( 1 );
	    }
	(void) sprintf( sayfilename, "%s/.sayfile", pw->pw_dir );
	sayfilefd = open( sayfilename, O_WRONLY|O_APPEND );
	if ( sayfilefd < 0 )
	    {
	    if ( errno != ENOENT ) /* don't bother reporting no .sayfile */
		(void) fprintf(
		    stderr, "%s: warning, couldn't open %s - %s\n", sw_argv0,
		    sayfilename, strerror( errno ) );
	    }
	else
	    {
	    if ( sw_lock( sayfilename, sayfilefd ) != 0 )
		(void) cleanup( 1 );
	    unlocksayfile = 1;
	    }

	if ( ( login = getlogin() ) == (char*) 0 )
	    if ( pw = getpwuid( getuid() ) )
		login = pw->pw_name;
	    else
		login = "???";
	if ( sw_savemine && strcmp( argv[1], login ) != 0 )
	    {
	    if ( ( home = getenv( "HOME" ) ) == (char*) 0 )
		{
		(void) fprintf(
		    stderr, "%s: can't find home directory\n", sw_argv0 );
		(void) cleanup( 1 );
		}
	    (void) sprintf( mysayfilename, "%s/.sayfile", home );
	    mysayfilefd = open( mysayfilename, O_WRONLY|O_APPEND );
	    if ( mysayfilefd < 0 )
		{
		if ( errno != ENOENT ) /* don't bother reporting no .sayfile */
		    (void) fprintf(
			stderr, "%s: warning, couldn't open %s - %s\n",
			sw_argv0, mysayfilename, strerror( errno ) );
		}
	    else
		{
		if ( sw_lock( mysayfilename, mysayfilefd ) != 0 )
		    (void) cleanup( 1 );
		unlockmysayfile = 1;
		}
	    }
	else
	    mysayfilefd = -1;

	if ( sayfilefd >= 0 || mysayfilefd >= 0 )
	    {
	    if ( ( fp = fopen( tmpfilename, "r" ) ) == (FILE*) 0 )
		{
		(void) fprintf(
		    stderr, "%s: couldn't re-open temp file %s - %s\n",
		    sw_argv0, tmpfilename, strerror( errno ) );
		(void) cleanup( 1 );
		}
	    /* Greeting code extracted from BSD write(1). */
	    if ( gethostname( host, sizeof(host) ) < 0 )
		(void) strcpy( host, "???" );
	    now = time( (time_t*) 0 );
	    nows = ctime( &now );
#ifdef WRITE_TIME_ONLY
	    nows[16] = '\0';
	    nows += 11;
#else /*WRITE_TIME_ONLY*/
	    nows[19] = '\0';
#endif /*WRITE_TIME_ONLY*/
	    /* Message separator - a control-A, a timestamp, and the sender. */
	    (void) sprintf( buf, "%c%d,%s\n", MSGSEP, (int) now, login );
	    if ( sayfilefd >= 0 )
		if ( write( sayfilefd, buf, strlen( buf ) ) < 0 )
		    {
		    (void) fprintf(
			stderr, "%s: error writing on %s - %s\n", sw_argv0,
			sayfilename, strerror( errno ) );
		    (void) cleanup( 1 );
		    }
	    if ( mysayfilefd >= 0 )
		if ( write( mysayfilefd, buf, strlen( buf ) ) < 0 )
		    {
		    (void) fprintf(
			stderr, "%s: error writing on %s - %s\n", sw_argv0,
			mysayfilename, strerror( errno ) );
		    (void) cleanup( 1 );
		    }
	    (void) sprintf( buf, WRITE_HEAD, login, host,
		ttyname( fileno( stderr ) ) + 5, nows );
	    if ( sayfilefd >= 0 )
		(void) write( sayfilefd, buf, strlen( buf ) );
	    if ( mysayfilefd >= 0 )
		(void) write( mysayfilefd, buf, strlen( buf ) );
	    /* And copy in the message. */
	    while ( fgets( buf, sizeof( buf ), fp ) != (char*) 0 )
		{
		if ( sayfilefd >= 0 )
		    (void) write( sayfilefd, buf, strlen( buf ) );
		if ( mysayfilefd >= 0 )
		    (void) write( mysayfilefd, buf, strlen( buf ) );
		}
	    if ( sayfilefd >= 0 )
		(void) write( sayfilefd, WRITE_TAIL, strlen(WRITE_TAIL) );
	    if ( mysayfilefd >= 0 )
		(void) write( mysayfilefd, WRITE_TAIL, strlen(WRITE_TAIL) );
	    if ( fclose( fp ) == EOF )
		{
		(void) fprintf(
		    stderr, "%s: error closing temp file %s - %s\n", sw_argv0,
		    tmpfilename, strerror( errno ) );
		(void) cleanup( 1 );
		}
	    if ( sayfilefd >= 0 )
		{
		unlocksayfile = 0;
		if ( sw_unlock( sayfilename, sayfilefd ) != 0 )
		    (void) cleanup( 1 );
		if ( close( sayfilefd ) < 0 )
		    {
		    (void) fprintf(
			stderr, "%s: error closing %s - %s\n", sw_argv0,
			sayfilename, strerror( errno ) );
		    (void) cleanup( 1 );
		    }
		}
	    if ( mysayfilefd >= 0 )
		{
		unlockmysayfile = 0;
		if ( sw_unlock( mysayfilename, mysayfilefd ) != 0 )
		    (void) cleanup( 1 );
		if ( close( mysayfilefd ) < 0 )
		    {
		    (void) fprintf(
			stderr, "%s: error closing %s - %s\n", sw_argv0,
			mysayfilename, strerror( errno ) );
		    (void) cleanup( 1 );
		    }
		}
	    }
	}
    else
	{
	/* Write(1) failed. */
	(void) printf(
	    "Your message could not be delivered for some reason.\n" );
	(void) printf( "I will mail it instead.\n" );
mail:
	(void) sprintf( buf, "mail -s 'this was a failing say' %s < %s",
	    argv[1], tmpfilename );
	(void) system( buf );
	}

    /* All done. */
    (void) cleanup( 0 );
    /*NOTREACHED*/
    }

static int
search_utmp( user )
    char* user;
    {
    struct utmp u;
    int ufd, found;
    char devtty[UT_LINESIZE+6];
    struct stat termstat;

    if ( ( ufd = open( _PATH_UTMP, O_RDONLY ) ) < 0 )
	{
	perror( _PATH_UTMP );
	(void) cleanup( 1 );
	}

    found = 0;
    while ( read( ufd, (char*) &u, sizeof(u) ) == sizeof(u) )
	{
	if ( strncmp( user, u.ut_name, sizeof(u.ut_name) ) == 0 )
	    {
	    found = 1;
	    (void) strcpy( devtty, "/dev/" );
	    (void) strncat( devtty, u.ut_line, UT_LINESIZE );
	    devtty[sizeof(devtty) - 1] = '\0';
	    if ( stat( devtty, &termstat ) >= 0 )
		{
		if ( ( termstat.st_mode & (S_IWRITE>>3) ) != 0 )
		    {
		    (void) close( ufd );
		    return SU_OK;
		    }
		}
	    }
	}
    (void) close( ufd );
    if ( found )
	return SU_MSGSOFF;
    return SU_NOTFOUND;
    }

static void
cleanup( status )
    int status;
    {
    sw_cleanup();
    if ( unlinktmpfile )
	(void) unlink( tmpfilename );
    if ( unlocksayfile )
	(void) sw_unlock( sayfilename, sayfilefd );
    if ( unlockmysayfile )
	(void) sw_unlock( mysayfilename, mysayfilefd );
    unlinktmpfile = 0;
    exit( status );
    }
