/* libsaywha.c - common routines for say(1) and wha(1)
**
** Copyright (C) 1990,1991 by Jef Poskanzer and Craig Leres.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/file.h>
#ifdef GROUP_WRITE
#include <grp.h>
#endif /*GROUP_WRITE*/
#ifdef LOCK_LOCKF
#include <unistd.h>
#endif /*LOCK_LOCKF*/
#include <errno.h>
#include "libsaywha.h"


/* Definitions. */

#ifdef GROUP_WRITE
#define SAYMODE 0620		/* GROUP_WRITE mode */
#else
#ifdef WORLD_READABLE
#define SAYMODE 0666		/* WORLD_READABLE mode */
#else
#define SAYMODE 0622		/* regular mode */
#endif /*WORLD_READABLE*/
#endif /*GROUP_WRITE*/

#ifdef LOCK_LINK
#define LOCK_FORMAT "%s.lock"
#endif /*LOCK_LINK*/
#define LOCK_TRIES 5
#define LOCK_SLEEP 1


/* Externals. */

extern int fprintf();
extern char* getenv();
extern char* mktemp();
extern char* strchr();
extern time_t time();


/* Public globals. */

char* sw_argv0;
int sw_keepdays = 3;
int sw_terse = 0;
#ifdef MAIL_EDITOR
int sw_mail_editor = 0;
#endif /*MAIL_EDITOR*/
int sw_savemine = 0;


/* Private globals. */

static char tmpfilename[] = "/tmp/sw.XXXXXX";
static int unlinktmpfile = 0;
static char sayfilename[200];
static FILE* sayfilefp;
static int unlocksayfile = 0;


/* Routines. */

int
sw_check_sayrc()
    {
    char* home;
    char buf[1000];
    char* cp;
    char* name;
    char* value;
    FILE* fp;

    if ( ( home = getenv( "HOME" ) ) == (char*) 0 )
	{
	(void) fprintf( stderr, "%s: can't find home directory\n", sw_argv0 );
	return -1;
	}
    (void) sprintf( buf, "%s/.sayrc", home );
    if ( ( fp = fopen( buf, "r" ) ) != (FILE*) 0 )
	{
	while ( fgets( buf, sizeof(buf), fp ) != (char*) 0 )
	    {
	    /* Trim comments. */
	    if ( ( cp = strchr( buf, '#' ) ) != (char*) 0 )
		*cp = '\0';
	    /* Trim trailing whitespace. */
	    for ( cp = &buf[strlen(buf) - 1];
		  *cp == ' ' || *cp == '\t' || *cp == '\n';
		  --cp )
		*cp = '\0';
	    /* Skip leading whitespace. */
	    for ( cp = buf;
		  *cp == ' ' || *cp == '\t' || *cp == '\n';
		  ++cp )
		;

	    /* Now, see what we've got. */
	    if ( *cp == '\0' )
		{
		/* Null command. */
		}
	    else if ( strncmp( cp, "set", 3 ) == 0 )
		{
		/* A set command. */
		cp += 3;
		while ( *cp != ' ' && *cp != '\t' && *cp != '\0' )
		    ++cp;
		while ( *cp != '\0' )
		    {
		    while ( *cp == ' ' || *cp == '\t' )
			++cp;
		    name = cp;
		    while ( *cp != '=' && *cp != ' ' && *cp != '\t' &&
			    *cp != '\0' )
			++cp;
		    if ( *cp == '=' )
			{
			/* set name=value */
			*cp++ = '\0';
			value = cp;
			while ( *cp != ' ' && *cp != '\t' && *cp != '\0' )
			    ++cp;
			*cp++ = '\0';
			}
		    else
			{
			/* set name */
			*cp++ = '\0';
			value = (char*) 0;
			}
		    /* Ok, we have name and value.  Now take a look. */
		    if ( strcmp( name, "keepdays" ) == 0 )
			{
			if ( value == (char*) 0 )
			    (void) fprintf( stderr,
			   "%s: missing value in .sayrc set keepdays command\n",
				sw_argv0 );
			else if ( sscanf( value, "%d", &sw_keepdays ) != 1 )
			    (void) fprintf( stderr,
	       "%s: non-numeric value in .sayrc set keepdays command: \"%s\"\n",
				sw_argv0, value );
			}
#ifdef MAIL_EDITOR
		    else if ( strcmp( name, "mail_editor" ) == 0 )
			{
			if ( value != (char*) 0 )
			    (void) fprintf( stderr,
	     "%s: extraneous value in .sayrc set mail_editor command: \"%s\"\n",
				sw_argv0, value );
			else
			    sw_mail_editor = 1;
			}
#endif /*MAIL_EDITOR*/
		    else if ( strcmp( name, "terse" ) == 0 )
			{
			if ( value != (char*) 0 )
			    (void) fprintf( stderr,
		   "%s: extraneous value in .sayrc set terse command: \"%s\"\n",
				sw_argv0, value );
			else
			    sw_terse = 1;
			}
		    else if ( strcmp( name, "savemine" ) == 0 )
			{
			if ( value != (char*) 0 )
			    (void) fprintf( stderr,
		"%s: extraneous value in .sayrc set savemine command: \"%s\"\n",
				sw_argv0, value );
			else
			    sw_savemine = 1;
			}
		    else
			(void) fprintf( stderr,
		    "%s: unrecognized variable in .sayrc set command: \"%s\"\n",
			    sw_argv0, name );
		    }
		}
	    else
		(void) fprintf( stderr,
		    "%s: unrecognized command in .sayrc: \"%s\"\n",
		    sw_argv0, cp );
	    }
	(void) fclose( fp );
#ifdef MAIL_EDITOR
	if ( sw_mail_editor && sw_terse )
	    (void) fprintf( stderr,
		"%s: set mail_editor and set terse don't make sense together\n",
		sw_argv0 );
#endif /*MAIL_EDITOR*/
	}
    return 0;
    }

int
sw_check_my_sayfile()
    {
    char* home;
    char buf[1000];
    FILE* fp2;
    int fd;
#ifdef GROUP_WRITE
    struct group *gp;
#endif /*GROUP_WRITE*/

    if ( ( home = getenv( "HOME" ) ) == (char*) 0 )
	{
	(void) fprintf( stderr, "%s: can't find home directory\n", sw_argv0 );
	return -1;
	}
    (void) sprintf( sayfilename, "%s/.sayfile", home );
    if ( ( sayfilefp = fopen( sayfilename, "r+" ) ) == (FILE*) 0 )
	{
	/* No .sayfile found - create one, with the proper protection. */
	if ( ( fd = open( sayfilename, O_CREAT | O_EXCL, 0 ) ) < 0 )
	    {
	    (void) fprintf(
		stderr, "%s: error creating .sayfile - %s\n",
		sw_argv0, strerror( errno ) );
	    return -1;
	    }
	if ( fchmod( fd, SAYMODE ) < 0 )
	    (void) fprintf(
		stderr, "%s: error changing mode on .sayfile - %s\n",
		sw_argv0, strerror( errno ) );
#ifdef GROUP_WRITE
	if ( ( gp = getgrnam( GROUP_WRITE ) ) == 0 )
	    (void) fprintf(
		stderr, "%s: can't find group \"%s\"\n",
		sw_argv0, GROUP_WRITE );
	else if ( fchown( fd, -1, gp->gr_gid ) < 0 )
	    (void) fprintf(
		stderr, "%s: error changing group on .sayfile - %s\n",
		sw_argv0, strerror( errno ) );
#endif /*GROUP_WRITE*/
	(void) close( fd );
	}
    else
	{
	/* Expire old messages. */
	int expired, saving;
	off_t newsize;
	time_t now;
	int then;
	char tuser[100];

	if ( sw_lock( sayfilename, fileno( sayfilefp ) ) != 0 )
	    return -1;
	unlocksayfile = 1;
	(void) mktemp( tmpfilename );
	if ( ( fp2 = fopen( tmpfilename, "w+" ) ) == (FILE*) 0 )
	    {
	    (void) fprintf(
		stderr, "%s: couldn't open temp file - %s\n",
		sw_argv0, strerror( errno ) );
	    return -1;
	    }
	unlinktmpfile = 1;
	(void) fchmod( fileno( fp2 ), 0600 );
	expired = 0;
	saving = 1;
	now = time( (time_t*) 0 );
	while ( fgets( buf, sizeof(buf), sayfilefp ) != (char*) 0 )
	    {
	    if ( buf[0] == MSGSEP )
		{
		if ( sscanf( buf + 1, "%d,%s\n", &then, tuser ) == 2 )
		    {
		    if ( then / 86400 + sw_keepdays > now / 86400 )
			{
			if ( ! expired )
			    break;	/* don't bother reading further */
			saving = 1;
			}
		    else
			{
			saving = 0;
			expired = 1;
			}
		    }
		else
		    {
		    saving = 0;
		    }
		}
	    if ( saving )
		(void) fputs( buf, fp2 );
	    }
	/* If any messages were expired, copy back the temp file. */
	if ( expired )
	    {
	    rewind( sayfilefp );
	    rewind( fp2 );
	    while ( fgets( buf, sizeof(buf), fp2 ) != (char*) 0 )
		(void) fputs( buf, sayfilefp );
	    newsize = ftell( sayfilefp );
	    }
	unlocksayfile = 0;
	if ( sw_unlock( sayfilename, fileno( sayfilefp ) ) != 0 )
	    return -1;
	if ( fclose( sayfilefp ) == EOF )
	    {
	    (void) fprintf(
		stderr, "%s: error closing %s - %s\n", sw_argv0, sayfilename,
		strerror( errno ) );
	    return -1;
	    }
	if ( fclose( fp2 ) == EOF )
	    {
	    (void) fprintf(
		stderr, "%s: error closing temp file %s - %s\n", sw_argv0,
		tmpfilename, strerror( errno ) );
	    return -1;
	    }
	(void) unlink( tmpfilename );
	unlinktmpfile = 0;
	/* The truncate should really go before we close. */
	if ( expired )
	    if ( truncate( sayfilename, newsize ) < 0 )
		{
		(void) fprintf(
		    stderr, "%s: error truncating %s - %s\n", sw_argv0,
		    sayfilename, strerror( errno ) );
		return -1;
		}
	}
    return 0;
    }

int
sw_lock( filename, fd )
    char* filename;
    int fd;
    {
#ifdef LOCK_LINK
    char lockfile[MAXPATHLEN];
#endif /*LOCK_LINK*/
    int try;

#ifdef LOCK_LINK
    (void) sprintf( lockfile, LOCK_FORMAT, filename );
#endif /*LOCK_LINK*/

    for ( try = LOCK_TRIES; try > 0; --try )
	{
#ifdef LOCK_LINK
	if ( link( filename, lockfile ) == 0 )
#endif /*LOCK_LINK*/
#ifdef LOCK_FLOCK
	if ( flock( fd, LOCK_EX | LOCK_NB ) == 0 )
#endif /*LOCK_FLOCK*/
#ifdef LOCK_LOCKF
	if ( lockf( fd, F_TLOCK, 1 ) == 0 )
#endif /*LOCK_LOCKF*/
	    return 0;

#ifdef LOCK_LINK
	if ( errno != EEXIST )
#endif /*LOCK_LINK*/
#ifdef LOCK_FLOCK
	if ( errno != EWOULDBLOCK )
#endif /*LOCK_FLOCK*/
#ifdef LOCK_LOCKF
	if ( errno != EAGAIN )
#endif /*LOCK_LOCKF*/
	    {
	    (void) fprintf(
		stderr, "%s: error trying to get lock on %s - %s\n",
		sw_argv0, filename, strerror( errno ) );
	    return -1;
	    }

	if ( try > 1 )
	    sleep( LOCK_SLEEP );
	}
    (void) fprintf(
        stderr, "%s: couldn't get lock on %s\n", sw_argv0, filename );
    return -1;
    }

int
sw_unlock( filename, fd )
    char* filename;
    int fd;
    {
#ifdef LOCK_LINK
    char lockfile[MAXPATHLEN];
#endif /*LOCK_LINK*/

#ifdef LOCK_LINK
    (void) sprintf( lockfile, LOCK_FORMAT, filename );
#endif /*LOCK_LINK*/

#ifdef LOCK_LINK
    if ( unlink( lockfile ) != 0 )
#endif /*LOCK_LINK*/
#ifdef LOCK_FLOCK
    if ( flock( fd, LOCK_UN ) != 0 )
#endif /*LOCK_FLOCK*/
#ifdef LOCK_LOCKF
    if ( lockf( fd, F_ULOCK, 1 ) != 0 )
#endif /*LOCK_LOCKF*/
	{
	(void) fprintf(
	    stderr, "%s: error trying to unlock %s - %s\n",
	    sw_argv0, filename, strerror( errno ) );
	return -1;
	}

    return 0;
    }

void
sw_cleanup()
    {
    if ( unlinktmpfile )
	{
	(void) unlink( tmpfilename );
	unlinktmpfile = 0;
	}
    if ( unlocksayfile )
	{
	(void) sw_unlock( sayfilename, fileno( sayfilefp ) );
	unlocksayfile = 0;
	}
    }
