/* hgrep - a front end to grep that highlights the string found
**
** Copyright (C) 1988,1989,1991 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdio.h>
#ifdef SYSV
#include <string.h>
#define index strchr
#define rindex strrchr
#else /*SYSV*/
#include <strings.h>
#endif /*SYSV*/

#define TBUFSIZE 1024
#define BIGBUFSIZE 10000

extern char* getenv();
extern char* tgetstr();

void putch();

int
main( argc, argv )
int argc;
char* argv[];
    {
    char* term;
    char* strptr;
    char* high_start;
    char* high_end;
    int dumb = 0;
    int iflag = 0;
    char buf[TBUFSIZE];
    static char strbuf[TBUFSIZE];
    char bigbuf[BIGBUFSIZE];
    char* grepname;
    char* grepword = (char*) 0;
    int pipefds[2], child;
    FILE* pipestream;
    int i, j, so_on, grepwordlen;
    unsigned char somap[BIGBUFSIZE];

    /* Initialize termcap stuff. */
    if ( isatty( fileno( stdout ) ) == 0 )
	dumb = 1;
    else
	{
	term = getenv( "TERM" );
	if ( term == 0 )
	    dumb = 1;
	else if ( tgetent( buf, term ) <= 0 )
	    dumb = 1;
	else
	    {
	    strptr = strbuf;
	    high_start = tgetstr( "md", &strptr );
	    if ( high_start != (char*) 0 )
		{
		high_end = tgetstr( "me", &strptr );
		if ( high_end == (char*) 0 )
		    dumb = 1;
		}
	    else
		{
		high_start = tgetstr( "so", &strptr );
		high_end = tgetstr( "se", &strptr );
		if ( high_start == (char*) 0 || high_end == (char*) 0 )
		    dumb = 1;
		}
	    }
	}

    /* Figure out which grep to use. */
    grepname = rindex( argv[0], '/' );
    if ( grepname == (char*) 0 )
	grepname = argv[0];
    else
	++grepname;
    if ( *grepname == 'h' )
	++grepname;
    if ( strcmp( grepname, "grep" ) != 0 &&
	 strcmp( grepname, "egrep" ) != 0 &&
	 strcmp( grepname, "fgrep" ) != 0 &&
	 strcmp( grepname, "ngrep" ) != 0 )
	grepname = "grep";	/* default to plain grep */
    argv[0] = grepname;

    /* Look through args for anything we must keep track of. */
    for ( i = 1; i < argc; ++i )
	{
	/* Check for -i flag. */
	if ( argv[i][0] == '-' && index( argv[i], 'i' ) != (char*) 0 )
	    iflag = 1;
	/* Check for -l flag. */
	if ( argv[i][0] == '-' && index( argv[i], 'l' ) != (char*) 0 )
	    dumb = 1;
	/* Check for -v flag. */
	if ( argv[i][0] == '-' && index( argv[i], 'v' ) != (char*) 0 )
	    dumb = 1;

	/* Save pointer to word we are grepping for - first non-flag arg. */
	if ( grepword == (char*) 0 && argv[i][0] != '-' )
	    {
	    grepword = argv[i];
	    if ( grepword[0] == '^' )
		++grepword;				/* hack hack */
	    grepwordlen = strlen( grepword );
	    if ( grepword[grepwordlen - 1] == '$' )
		{
		grepword[grepwordlen - 1] = '\0';	/* another hack hack */
		--grepwordlen;
		}
	    }
	}

    if ( dumb )
	{
	execvp( grepname, argv );
	perror( "execvp" );
	exit( 1 );
	}

    /* Spawn the grep. */
    if ( pipe( pipefds ) == -1 )
	{
	perror( "hgrep pipe" );
	exit( -1 );
	}
    child = fork( );
    if ( child == -1 )
	{
	perror( "hgrep fork" );
	exit( 1 );
	}
    else if ( child == 0 )
	{
	(void) close( pipefds[0] );	/* close read end of pipe */
	(void) dup2( pipefds[1], 1 );	/* replace stdout w/write end of pipe */
	(void) close(pipefds[1] );	/* and close it */
	execvp( grepname, argv );
	perror( "execvp" );
	exit( 1 );
	}
    (void) close( pipefds[1] );		/* close write end of pipe */
    pipestream = fdopen( pipefds[0], "r" );
    if ( pipestream == (FILE*) 0 )
	{
	perror( "hgrep fdopen" );
	exit( 1 );
	}

    /* Now read and handle results of grep. */
    so_on = 0;
    while ( fgets( bigbuf, BIGBUFSIZE, pipestream ) != (char*) 0 )
	{
	/* Figure out what to highlight.  This is fairly inefficient,
	** but since we are operating on the output of grep and not
	** the input, it's ok. */
	for ( i = 0; bigbuf[i] != '\0'; ++i )
	    somap[i] = 0;
	for ( i = 0; bigbuf[i] != '\0'; ++i )
	    if ( iflag )
		{
		if ( cistrncmp( &(bigbuf[i]), grepword, grepwordlen ) == 0 )
		    for ( j = i; j < i + grepwordlen; ++j )
			somap[j] = 1;
		}
	    else
		{
		if ( strncmp( &(bigbuf[i]), grepword, grepwordlen ) == 0 )
		    for ( j = i; j < i + grepwordlen; ++j )
			somap[j] = 1;
		}

	/* And now do the highlighting. */
	for ( i = 0; bigbuf[i] != '\0'; ++i )
	    {
	    if ( somap[i] )
		{
		if ( ! so_on )
		    {
		    tputs( high_start, 1, putch );
		    so_on = 1;
		    }
		}
	    else
		{
		if ( so_on )
		    {
		    tputs( high_end, 1, putch );
		    so_on = 0;
		    }
		}
	    putchar( bigbuf[i] );
	    }
	}

    /* Bye. */
    if ( so_on )
	tputs( high_end, 1, putch );
    (void) fclose( pipestream );
    exit( 0 );
    }

void
putch( ch )
char ch;
    {
    putchar( ch );
    }


/* cistrncmp - case-insensitive version of strncmp */

#include <ctype.h>

int
cistrncmp( s1, s2, n )
char* s1;
char* s2;
int n;
    {
    while ( --n >= 0 && (isupper(*s1)?tolower(*s1):*s1) == (isupper(*s2)?tolower(*s2++):*s2++) )
	if ( *s1++ == '\0' )
	    return 0;
    return n < 0 ? 0 : *s1 - *--s2;
    }
