/* graymilter - simple graylist mail filter module
**
** Copyright  2004 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/

#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_GRP_H
#include <grp.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#include "libmilter/mfapi.h"

#include "version.h"
#include "iparray.h"


/* Defines. */

#define MIN_UPDATE_INTERVAL 10	/* min seconds between updates */
#define MIN_FILE_AGE 30		/* min age of changed file before reading */

#define DEFAULT_GRAYTIME 1500            /* 25 minutes */
#define DEFAULT_WHITETIME 172800         /* two days */
#define N_LISTS 10

#define min(a,b) ((a) < (b) ? (a) : (b))
#define max(a,b) ((a) > (b) ? (a) : (b))


/* Forwards. */

static void usage( void );

static void init_uid( const char* user );
static void init_socket( const char* sockarg );
static void fini_socket( const char* sockarg );
static void unlink_socket( const char* sockarg );
static void init_daemon( void );
static void init_pidfile( const char* pidfilename );
static void fini_pidfile( const char* pidfilename );
static void init_iparrays( void );
static void fini_iparrays( void );

static int stat_file( time_t current_time, time_t mtime, char* filename );
static time_t read_file( iparray list, char* filename );
static void trim( char* str );
static void update( void );

/* The milter callback routines.  Signatures for these are fixed
** by the milter API.
*/
static sfsistat gray_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr );
static sfsistat gray_envfrom( SMFICTX* ctx, char** fromargs );
static sfsistat gray_close( SMFICTX* ctx );


/* Globals. */

static char* argv0;
static int graytime, whitetime;
static char* initial_whitelist_filename;
static time_t initial_whitelist_mtime;
static iparray initial_whitelist;
static octets* graylists[N_LISTS];
static int max_graylists[N_LISTS];
static int n_graylists[N_LISTS];
static iparray whitelists[N_LISTS];
static int n_whitelists[N_LISTS];
static octets localhost = { 127, 0, 0, 1 };
static int update_interval, gray_update_interval, white_update_interval;
static time_t last_update, last_gray_update, last_white_update;
static pthread_mutex_t lock;

static struct smfiDesc smfilter =
    {
    "GRAY",		/* filter name */
    SMFI_VERSION,	/* version code -- do not change */
    0,			/* flags */
    gray_connect,	/* connection info filter */
    NULL,		/* SMTP HELO command filter */
    gray_envfrom,	/* envelope sender filter */
    NULL,		/* envelope recipient filter */
    NULL,		/* header filter */
    NULL,		/* end of header */
    NULL,		/* body block filter */
    NULL,		/* end of message */
    NULL,		/* message aborted */
    gray_close		/* connection cleanup */
    };


int
main( int argc, char** argv )
    {
    int argn;
    char* user;
    char* pidfilename;
    int nodaemon;
    char* sockarg;

    /* Figure out the program's name. */
    argv0 = strrchr( argv[0], '/' );
    if ( argv0 != (char*) 0 )
	++argv0;
    else
	argv0 = argv[0];

    /* Parse args. */
    graytime = DEFAULT_GRAYTIME;
    whitetime = DEFAULT_WHITETIME;
    initial_whitelist_filename = (char*) 0;
    user = (char*) 0;
    pidfilename = (char*) 0;
    nodaemon = 0;
    argn = 1;
    while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
	{
	if ( strncmp( argv[argn], "-graytime", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    graytime = atoi( argv[argn] );
	    }
	else if ( strncmp( argv[argn], "-whitetime", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    whitetime = atoi( argv[argn] );
	    }
	else if ( strncmp( argv[argn], "-initialwhitelist", max( strlen( argv[argn] ), 3 ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    initial_whitelist_filename = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-user", max( strlen( argv[argn] ), 3 ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    user = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-pidfile", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    pidfilename = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-nodaemon", strlen( argv[argn] ) ) == 0 )
	    nodaemon = 1;
	else if ( strncmp( argv[argn], "-X", strlen( argv[argn] ) ) == 0 )
	    nodaemon = 1;
	else
	    usage();
	++argn;
	}
    if ( argn >= argc )
	usage();
    sockarg = argv[argn++];
    if ( argn != argc )
	usage();
    if ( graytime < 1 || whitetime < 1 || whitetime < graytime )
	usage();

    openlog( argv0, 0, LOG_MAIL );
    init_uid( user );
    init_socket( sockarg );
    if ( ! nodaemon )
	init_daemon();
    init_pidfile( pidfilename );
    init_iparrays();

    if ( pthread_mutex_init( &lock, (pthread_mutexattr_t*) 0 ) != 0 )
	{
	(void) fprintf( stderr, "%s: pthread_mutex_init failed\n", argv0 );
	exit( 1 );
	}

    gray_update_interval = graytime / N_LISTS;
    white_update_interval = whitetime / N_LISTS;
    if ( initial_whitelist_filename != (char*) 0 )
	update_interval = MIN_UPDATE_INTERVAL;
    else
	update_interval = min( gray_update_interval, white_update_interval ) / 2;
    last_update = last_gray_update = last_white_update = time( (time_t*) 0 );

    if ( smfi_main() == MI_FAILURE )
	{
	syslog( LOG_ERR, "smfi_main() failed" );
	exit( 1 );
	}

    (void) pthread_mutex_destroy( &lock );
    fini_iparrays();
    fini_pidfile( pidfilename );
    fini_socket( sockarg );
    closelog();
    exit( 0 );
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage:  %s [-graytime seconds] [-whitetime seconds] [-initialwhitelist file] [-user user] [-pidfile filename] [-nodaemon|-X] socket\n", argv0 );
    exit( 1 );
    }


static void
init_uid( const char* user )
    {
    struct passwd* pwd;
    char* ep;
    int uid;

    if ( getuid() == 0 )
	{
	/* If we're root, the very first thing we do is become another user. */
	if ( user == (char*) 0 )
	    (void) fprintf( stderr, "%s: warning: started as root but no --user flag specified\n", argv0 );
	else
	    {
	    /* Is it a number? */
	    uid = strtol( user, &ep, 0 );
	    if ( *ep == '\0' )
		pwd = getpwuid( uid );
	    else
		pwd = getpwnam( user );
	    if ( pwd == (struct passwd*) 0 )
		{
		(void) fprintf( stderr, "%s: unknown user: '%s'\n", argv0, user );
		exit( 1 );
		}
	    /* Set aux groups to null. */
	    if ( setgroups( 0, (gid_t*) 0 ) < 0 )
		{
		perror( "setgroups" );
		exit( 1 );
		}
	    /* Set primary group. */
	    if ( setgid( pwd->pw_gid ) < 0 )
		{
		perror( "setgid" );
		exit( 1 );
		}

	    /* Try setting aux groups correctly - not critical if this fails. */
	    if ( initgroups( user, pwd->pw_gid ) < 0 )
		    perror( "initgroups" );
	    /* Set uid. */
	    if ( setuid( pwd->pw_uid ) < 0 )
		{
		perror( "setuid" );
		exit( 1 );
		}
	    }
	}
    else
	{
	/* If we're not root but got a -user flag anyway, that's an error. */
	if ( user != (char*) 0 )
	    {
	    (void) fprintf( stderr, "%s: can't switch users if not started as root\n", argv0 );
	    exit( 1 );
	    }
	}
    }


static void
init_socket( const char* sockarg )
    {
    unlink_socket( sockarg );

    /* Harden our umask so that the new socket gets created securely. */
    umask( 0077 );

    /* Initialize milter stuff. */
    smfi_setconn( (char*) sockarg );
    if ( smfi_register( smfilter ) == MI_FAILURE )
	{
	(void) fprintf( stderr, "%s: smfi_register() failed\n", argv0 );
	exit( 1 );
	}
    }


static void
fini_socket( const char* sockarg )
    {
    unlink_socket( sockarg );
    }


static void
unlink_socket( const char* sockarg )
    {
    /* Remove old local socket.  Actually we should probably stat
    ** the file and check that it is in fact a socket before removing
    ** it.
    */
    if ( strncasecmp( sockarg, "unix:", 5 ) == 0 )
	{
	if ( unlink( &sockarg[5] ) < 0 )
	    if ( errno != ENOENT )
		perror( &sockarg[5] );
	}
    else if ( strncasecmp( sockarg, "local:", 6 ) == 0 )
	{
	if ( unlink( &sockarg[6] ) < 0 )
	    if ( errno != ENOENT )
		perror( &sockarg[6] );
	}
    else if ( strchr( sockarg, ':' ) == (char*) 0 )
	{
	if ( unlink( sockarg ) < 0 )
	    if ( errno != ENOENT )
		perror( sockarg );
	}
    }


static void
init_daemon( void )
    {
    /* Daemonize. */
#ifdef HAVE_DAEMON
    if ( daemon( 0, 0 ) < 0)
	{
	perror( "daemon" );
	exit( 1 );
	}
#else /* HAVE_DAEMON */
    switch ( fork() )
	{
	case 0:
	    break;  
	case -1:
	    syslog( LOG_CRIT, "fork: %m" );
	    perror( "fork" );
	    exit( 1 );
	default:
	    exit( 0 );
	}
#ifdef HAVE_SETSID
    setsid();
#endif /* HAVE_SETSID */
#endif /* HAVE_DAEMON */
    }


static void
init_pidfile( const char* pidfilename )
    {
    if ( pidfilename != (char*) 0 )
	{
	FILE* fp;

	fp = fopen( pidfilename, "w" );
	if ( fp == (FILE*) 0 )
	    syslog( LOG_ERR, "unable to write PID file - %m" );
	else
	    {
	    (void) fprintf( fp, "%ld\n", (long) getpid() );
	    (void) fclose( fp );
	    }
	}
    }


static void
fini_pidfile( const char* pidfilename )
    {
    if ( pidfilename != (char*) 0 )
	(void) unlink( pidfilename );
    }


static void
init_iparrays( void )
    {       
    int i;

    if ( initial_whitelist_filename != (char*) 0 )
	{
	initial_whitelist = iparray_new();
	if ( initial_whitelist == (iparray) 0 )
	    {
	    (void) fprintf( stderr, "%s: whitelist create failed\n", argv0 );
	    exit( 1 );
	    }
	initial_whitelist_mtime = read_file( initial_whitelist, initial_whitelist_filename );
	}

    for ( i = 0; i < N_LISTS; ++i )
	{
	max_graylists[i] = 1000;	/* arbitrary */
	graylists[i] = (octets*) malloc( max_graylists[i] * sizeof(octets) );
	if ( graylists[i] == (octets*) 0 )
	    {
	    (void) fprintf( stderr, "%s: graylist create failed\n", argv0 );
	    exit( 1 );
	    }
	n_graylists[i] = 0;
	whitelists[i] = iparray_new();
	if ( whitelists[i] == (iparray) 0 )
	    {
	    (void) fprintf( stderr, "%s: whitelist create failed\n", argv0 );
	    exit( 1 );
	    }
	n_whitelists[i] = 0;
	}
    }


static void
fini_iparrays( void )
    {
    int i;

    for ( i = 0; i < N_LISTS; ++i )
	{
	free( (void*) graylists[i] );
	iparray_delete( whitelists[i] );
	}
    if ( initial_whitelist_filename != (char*) 0 )
	iparray_delete( initial_whitelist );
    iparray_fini();
    }


/* Returns 1 if the file is still current, else 0. */
static int
stat_file( time_t current_time, time_t mtime, char* filename )
    {
    struct stat sb;

    if ( stat( filename, &sb ) < 0 )
	return 1;	/* Can't stat it?  Ignore. */
    if ( sb.st_mtime == mtime )
	return 1;	/* Unchanged. */
    if ( current_time - sb.st_mtime < MIN_FILE_AGE )
	return 1;	/* Not old enough, we'll catch it next time. */
    return 0;		/* Changed. */
    }


/* Reads one file into the database. */
static time_t
read_file( iparray list, char* filename )
    {
    FILE* fp;
    struct stat sb;
    time_t mtime;
    char line[10000];
    octets ip;

    syslog( LOG_INFO, "reading %s", filename );
    fp = fopen( filename, "r" );
    if ( fp == (FILE*) 0 )
	{
	syslog( LOG_ERR, "%s - %m", filename );
	mtime = (time_t) -1;
	}
    else
	{
	if ( fstat( fileno(fp), &sb ) == 0 )
	    mtime = sb.st_mtime;
	else
	    mtime = (time_t) -1;
	while ( fgets( line, sizeof(line), fp ) != (char*) 0 )
	    {
	    trim( line );
	    if ( line[0] == '\0' )
		continue;
	    if ( iparray_parse_octets( line, &ip ) )
		(void) iparray_incr( list, ip );
	    else
		syslog( LOG_INFO, "unparsable IP address - \"%s\"", line );
	    }
	(void) fclose( fp );
	}
    return mtime;
    }


static void
trim( char* str )
    {
    char* cp;
    int len;

    cp = strchr( str, '#' );
    if ( cp != (char*) 0 )
	*cp = '\0';
    len = strlen( str );
    while ( str[len-1] == '\n' || str[len-1] == '\r' || str[len-1] == ' ' || str[len-1] == '\t' )
	{
	--len;
	str[len] = '\0';
	}
    }


static void
update( void )
    {
    time_t current_time;
    int i;
    octets* tmp_graylist;
    int tmp_max_graylist;
    iparray tmp_whitelist;

    current_time = time( (time_t*) 0 );
    if ( current_time - last_update < update_interval )
	return;
    last_update = current_time;

    if ( pthread_mutex_lock( &lock ) == 0 )
	{
	if ( initial_whitelist_filename != (char*) 0 )
	    if ( ! stat_file( current_time, initial_whitelist_mtime, initial_whitelist_filename ) )
		{
		syslog( LOG_INFO, "initial whitelist file changed - autoupdating" );
		iparray_clear( initial_whitelist );
		initial_whitelist_mtime = read_file( initial_whitelist, initial_whitelist_filename );
		}

	while ( current_time - last_gray_update >= gray_update_interval )
	    {
	    /* Graduate addresses from the last graylist to the first whitelist. */
	    syslog( LOG_INFO, "graduating %d addresses to whitelist", n_graylists[N_LISTS-1] );
	    for ( i = 0; i < n_graylists[N_LISTS-1]; ++i )
		(void) iparray_incr( whitelists[0], graylists[N_LISTS-1][i] );
	    n_whitelists[0] += n_graylists[N_LISTS-1];
	    /* Rotate the graylists. */
	    tmp_graylist = graylists[N_LISTS-1];
	    tmp_max_graylist = max_graylists[N_LISTS-1];
	    for ( i = N_LISTS - 1; i > 0; --i )
		{
		graylists[i] = graylists[i-1];
		max_graylists[i] = max_graylists[i-1];
		n_graylists[i] = n_graylists[i-1];
		}
	    graylists[0] = tmp_graylist;
	    max_graylists[0] = tmp_max_graylist;
	    /* Clear out the first graylist. */
	    n_graylists[0] = 0;
	    /* Bump update time. */
	    last_gray_update += gray_update_interval;
	    }

	while ( current_time - last_white_update >= white_update_interval )
	    {
	    syslog( LOG_INFO, "expiring %d addresses from whitelist", n_whitelists[N_LISTS-1] );
	    /* Rotate the whitelists. */
	    tmp_whitelist = whitelists[N_LISTS-1];
	    for ( i = N_LISTS - 1; i > 0; --i )
		{
		whitelists[i] = whitelists[i-1];
		n_whitelists[i] = n_whitelists[i-1];
		}
	    whitelists[0] = tmp_whitelist;
	    /* Clear out the first whitelist. */
	    iparray_clear( whitelists[0] );
	    n_whitelists[0] = 0;
	    /* Bump update time. */
	    last_white_update += white_update_interval;
	    }

	(void) pthread_mutex_unlock( &lock );
	}
    }


/* The private data struct. */
struct connection_data {
    octets ip;
    int action;
    char *connhost;
    };
#define ACTION_UNKNOWN 0
#define ACTION_TEMPFAIL 1


/* gray_connect - handle the initial TCP connection
**
** Called at the start of a connection.  Any per-connection data should
** be initialized here.
**
** connhost: The hostname of the client, based on a reverse lookup.
** connaddr: The client's IP address, based on getpeername().
*/
static sfsistat
gray_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr )
    {
    struct connection_data* cd;
    struct sockaddr_in* sin;
    char* char_addr;
    int i;

    update();

    if ( connaddr == (_SOCK_ADDR*) 0 )
	return SMFIS_ACCEPT;	/* can't deal with it */

    cd = (struct connection_data*) malloc( sizeof(struct connection_data) );
    if ( cd == (struct connection_data*) 0 )
	return SMFIS_TEMPFAIL;
    (void) smfi_setpriv( ctx, (void*) cd );
    cd->action = ACTION_UNKNOWN;

    sin = (struct sockaddr_in*) connaddr;

    char_addr = (char*) &sin->sin_addr.s_addr;
    cd->ip.a = char_addr[0];
    cd->ip.b = char_addr[1];
    cd->ip.c = char_addr[2];
    cd->ip.d = char_addr[3];
    cd->ip.w = 32;

    cd->connhost = strdup( connhost );	/* don't care about failure */

    /* Add the address to the first graylist. */
    if ( n_graylists[0] >= max_graylists[0] )
	{
	/* Oosp, got to expand the list. */
	if ( pthread_mutex_lock( &lock ) != 0 )
	    return SMFIS_TEMPFAIL;
	max_graylists[0] *= 2;
	syslog( LOG_NOTICE, "expanding a graylist to %d items", max_graylists[0] );
	graylists[0] = (octets*) realloc( (void*) graylists[0], max_graylists[0] * sizeof(octets) );
	(void) pthread_mutex_unlock( &lock );
	}
    graylists[0][n_graylists[0]] = cd->ip;
    ++n_graylists[0];

    /* Special case for localhost. */
    if ( cd->ip.a == localhost.a )	/* localhost is actually a class-A network */
	{
	syslog( LOG_INFO, "%d.%d.%d.%d is localhost - accepting", cd->ip.a, cd->ip.b, cd->ip.c, cd->ip.d );
	return SMFIS_ACCEPT;
	}

    /* Check if the address is in the optional initial whitelist. */
    if ( initial_whitelist_filename != (char*) 0 )
	if ( iparray_get( initial_whitelist, cd->ip ) > 0 )
	    {
	    syslog( LOG_INFO, "%d.%d.%d.%d (%s) is whitelisted - accepting", cd->ip.a, cd->ip.b, cd->ip.c, cd->ip.d, cd->connhost );
	    return SMFIS_ACCEPT;
	    }

    /* Check if the address is in any of the whitelists. */
    for ( i = 0; i < N_LISTS; ++i )
	{
	if ( iparray_get( whitelists[i], cd->ip ) > 0 )
	    {
	    syslog( LOG_INFO, "%d.%d.%d.%d (%s) has been seen before - accepting", cd->ip.a, cd->ip.b, cd->ip.c, cd->ip.d, cd->connhost );
	    return SMFIS_ACCEPT;
	    }
	}

    /* Tempfail it.  But we can't actually send the response yet, because
    ** we're not allowed to call smfi_setreply() from the _connect
    ** callback.
    */
    cd->action = ACTION_TEMPFAIL;
    return SMFIS_CONTINUE;
    }


/* gray_envfrom - handle the MAIL FROM:<> command
**
** Called at the start of each message.  There may be multiple messages
** in a connection.  Any per-message data should be initialized here.
**
** fromargs: Null-terminated SMTP command arguments; fromargs[0] is the
**                 sender address.
*/
static sfsistat
gray_envfrom( SMFICTX* ctx, char** fromargs )
    {
    struct connection_data* cd = (struct connection_data*) smfi_getpriv( ctx );
    char* auth_type;
    char* verify;

    /* The temporary failure response can't happen in the connect handler
    ** because we haven't started speaking SMTP protocol yet.  You need to
    ** speak SMTP in order to send back an SMTP error response.
    **
    ** It used to happen in the HELO handler, which is the next one to run.
    **
    ** Now it is here in the MAIL FROM handler, because this is the first
    ** place where we can check the authentication macros.
    */

    /* Authenticated messages get accepted without question. */
    auth_type = smfi_getsymval( ctx, "{auth_type}" );
    verify = smfi_getsymval( ctx, "{verify}" );
    if ( auth_type != (char*) 0 ||
         ( verify != (char*) 0 && strcmp( verify, "OK" ) == 0 ) )
	{
	syslog( LOG_INFO, "%d.%d.%d.%d (%s) is authenticated - accepting", cd->ip.a, cd->ip.b, cd->ip.c, cd->ip.d, cd->connhost );
	return SMFIS_ACCEPT;
	}

    if ( cd->action == ACTION_TEMPFAIL )
	{
	/* Send back the temporary failure code. */
	/* syslog( LOG_INFO, "%d.%d.%d.%d (%s) is new - tempfailing", cd->ip.a, cd->ip.b, cd->ip.c, cd->ip.d, cd->connhost ); */
	(void) smfi_setreply( ctx, "421", "4.3.2", "graylisted - please try again later" );
	return SMFIS_TEMPFAIL;
	}

    return SMFIS_CONTINUE;
    }


/* gray_close - handle the connection being closed
**
** Called once at the end of a connection.  Any per-connection data
** should be freed here.
*/
static sfsistat
gray_close( SMFICTX* ctx )
    {
    struct connection_data* cd = (struct connection_data*) smfi_getpriv( ctx );

    if ( cd != (struct connection_data*) 0 )
	{
	(void) smfi_setpriv( ctx, (void*) 0 );
	if ( cd->connhost != (char*) 0 )
	    free( cd->connhost );
	free( (void*) cd );
	}

    return SMFIS_CONTINUE;
    }
