/* libdb.c - library for simple database manager
**
** Copyright  2009 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/

#include <fcntl.h>
#include <db.h>
#include <libdb.h>


#define min(a,b) ((a)<(b)?(a):(b))


void*
db_openread( char* filename )
    {
    return (void*) dbopen( filename, O_RDONLY|O_SHLOCK, 0644, DB_HASH, (struct HASHINFO*) 0 );
    }


void*
db_openwrite( char* filename )
    {
    return (void*) dbopen( filename, O_CREAT|O_RDWR|O_EXLOCK, 0644, DB_HASH, (struct HASHINFO*) 0 );
    }


int
db_get( void* opaque_db, char* key, char* value, size_t value_size )
    {
    DB* db;
    DBT key_dbt;
    DBT value_dbt;
    int r;
    size_t l;

    db = (DB*) opaque_db;
    key_dbt.data = (void*) key;
    key_dbt.size = strlen( key );
    r = db->get( db, &key_dbt, &value_dbt, 0 );
    if ( r == 0 )
	{
	l = min( value_size - 1, value_dbt.size );
	(void) memcpy( value, value_dbt.data, l );
	value[l] = '\0';
	}
    return r;
    }


int
db_next( void* opaque_db, char* key, size_t key_size, char* value, size_t value_size )
    {
    DB* db;
    DBT key_dbt;
    DBT value_dbt;
    int r;
    size_t l;

    db = (DB*) opaque_db;
    r = db->seq( db, &key_dbt, &value_dbt, R_NEXT );      
    if ( r == 0 )
	{
	l = min( key_size - 1, key_dbt.size );
	(void) memcpy( key, key_dbt.data, l );
	key[l] = '\0';
	l = min( value_size - 1, value_dbt.size );
	(void) memcpy( value, value_dbt.data, l );
	value[l] = '\0';
	}
    return r;
    }


int
db_put( void* opaque_db, char* key, char* value )
    {
    DB* db;
    DBT key_dbt;
    DBT value_dbt;

    db = (DB*) opaque_db;
    key_dbt.data = (void*) key;
    key_dbt.size = strlen( key );
    value_dbt.data = (void*) value;
    value_dbt.size = strlen( value );
    return db->put( db, &key_dbt, &value_dbt, 0 );
    }


int
db_del( void* opaque_db, char* key )
    {
    DB* db;
    DBT key_dbt;

    db = (DB*) opaque_db;
    key_dbt.data = (void*) key;
    key_dbt.size = strlen( key );
    return db->del( db, &key_dbt, 0 );
    }


int
db_close( void* opaque_db )
    {
    DB* db;

    db = (DB*) opaque_db;
    return db->close( db );
    }
