/* db.c - simple database manager
**
** Copyright  2009 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>
#include <limits.h>

#include <libdb.h>


int
main( int argc, char* argv[] )
    {
    int delete_flag, write_flag, read_flag;
    int writeable;
    char* filename;
    char* key;
    char* value;
    void* db;
    int r;
    char line[10000];
    size_t len;
    char* tab;
    char* usage = "db [-dwr] database key [value]\n";

    delete_flag = write_flag = read_flag = 0;
    writeable = 0;

    if ( strcmp( argv[1], "-d" ) == 0 )
	{
	if ( argc == 4 )
	    {
	    delete_flag = 1;
	    filename = argv[2];
	    key = argv[3];
	    value = (char*) 0;
	    writeable = 1;
	    }
	else
	    {
	    (void) fprintf( stderr, usage );
	    exit( 1 );
	    }
	}
    else if ( strcmp( argv[1], "-w" ) == 0 )
	{
	if ( argc == 3 )
	    {
	    write_flag = 1;
	    filename = argv[2];
	    key = (char*) 0;
	    value = (char*) 0;
	    }
	else
	    {
	    (void) fprintf( stderr, usage );
	    exit( 1 );
	    }
	}
    else if ( strcmp( argv[1], "-r" ) == 0 )
	{
	if ( argc == 3 )
	    {
	    read_flag = 1;
	    filename = argv[2];
	    key = (char*) 0;
	    value = (char*) 0;
	    writeable = 1;
	    }
	else
	    {
	    (void) fprintf( stderr, usage );
	    exit( 1 );
	    }
	}
    else if ( argv[1][0] == '-' )
	{
	(void) fprintf( stderr, usage );
	exit( 1 );
	}
    else if ( argc == 3 )
	{
	filename = argv[1];
	key = argv[2];
	value = (char*) 0;
	}
    else if ( argc == 4 )
	{
	filename = argv[1];
	key = argv[2];
	value = argv[3];
	writeable = 1;
	}
    else
	{
	(void) fprintf( stderr, usage );
	exit( 1 );
	}

    if ( writeable )
	db = db_openwrite( filename );
    else
	db = db_openread( filename );
    if ( db == (void*) 0 )
	{
	perror( "db open" );
	exit( 1 );
	}

    if ( write_flag )
	{
	for (;;)
	    {
	    char k[10000], v[10000];
	    r = db_next( db, k, sizeof(k), v, sizeof(v) );
	    if ( r == -1 )
		{
		perror( "db seq" );
		exit( 1 );
		}
	    if ( r == 1 )
		break;
	    (void) printf( "%s\t%s\n", k, v );
	    }
	}
    else if ( read_flag )
	{
	while ( fgets( line, sizeof(line), stdin ) != (char*) 0 )
	    {
	    len = strlen( line );
	    if ( len > 0 && line[len - 1] == '\n' )
		line[--len] = '\0';
	    tab = strchr( line, '\t' );
	    if ( tab == (char*) 0 )
		(void) fprintf( stderr, "db: no tab separator found in line '%s'\n", line );
	    else
		{
		*tab++ = '\0';
		if ( db_put( db, line, tab ) != 0 )
		    perror( "db put" );
		}
	    }
	}
    else
	{
	if ( delete_flag )
	    {
	    r = db_del( db, key );
	    if ( r == -1 )
		{
		perror( "db del" );
		exit( 1 );
		}
	    if ( r == 1 )
		{
		(void) fprintf( stderr, "db: key not found\n" );
		exit( 1 );
		}
	    }
	else if ( value == (char*) 0 )
	    {
	    char v[10000];
	    r = db_get( db, key, v, sizeof(v) );
	    if ( r == -1 )
		{
		perror( "db get" );
		exit( 1 );
		}
	    if ( r == 1 )
		{
		(void) fprintf( stderr, "db: key not found\n" );
		exit( 1 );
		}
	    (void) printf( "%s\n", v );
	    }
	else
	    {
	    if ( db_put( db, key, value ) != 0 )
		{
		perror( "db put" );
		exit( 1 );
		}
	    }
	}

    if ( db_close( db ) != 0 )
	{
	perror( "db close" );
	exit( 1 );
	}

    exit( 0 );
    }
