/* timegraph.c - graph or histogram time-based data
**
** Copyright (C) 1995,1999,2000,2001 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>

#include "date_parse.h"


#define DEFAULT_LINES 10

#define INTERVAL_SLOP 0.02

#define IM_GRAPH 0
#define IM_HIST 1

#define OM_HOUR		0
#define OM_TWOHOURS	1
#define OM_FOURHOURS	2
#define OM_SIXHOURS	3
#define OM_DAY		4
#define OM_TWODAYS	5
#define OM_THREEDAYS	6
#define OM_FOURDAYS	7
#define OM_WEEK		8
#define OM_TWOWEEKS	9
#define OM_MONTH	10
#define OM_TWOMONTHS	11
#define OM_THREEMONTHS	12
#define OM_SIXMONTHS	13
#define OM_YEAR		14
#define OM_TWOYEARS	15
#define OM_THREEYEARS	16
#define OM_FIVEYEARS	17
#define OM_SIXYEARS	18
#define OM_TENYEARS	19
#define OM_LAST	OM_TENYEARS

#define MAXBINS 80

static int nbins_tab[] = {
/*  HOUR TWOHOURS FOURHOURS SIXHOURS DAY   2DAY    3DAY    4DAY  WEEK  2WEEK */
    60,  60,      4*18,     6*12,    24*3, 12*3*2, 12*2*3, 18*4, 7*10, 14*5,
/*  MONTH 2MONTH 3MONTH 6MONTH YEAR  2YEAR 3YEAR 5YEAR 6YEAR 10YEAR */
    31*2, 62,    3*24,  6*12,  12*6, 24*3, 36*2, 5*12, 6*12, 10*7 };

static char* timelabel[] = {
    "%02d00 %02d05 %02d10 %02d15 %02d20 %02d25 %02d30 %02d35 %02d40 %02d45 %02d50 %02d55 \n",
    "%02d00 %02d10 %02d20 %02d30 %02d40 %02d50|%02d00 %02d10 %02d20 %02d30 %02d40 %02d50 \n",
    "%02d:10:20:30:40:50|%02d:10:20:30:40:50|%02d:10:20:30:40:50|%02d:10:20:30:40:50 \n",
    "   %02d:00   |   %02d:00   |   %02d:00   |   %02d:00   |   %02d:00   |   %02d:00    \n",
    "00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16 17 18 19 20 21 22 23 \n",
    "00 02 04 06 08 10 %s 14 16 18 20 22|00 02 04 06 08 10 %s 14 16 18 20 22 \n",
    "01234567890%s3456789012|01234567890%s3456789012|01234567890%s34567890123\n",
    "01356801%s3567902|01356801%s3567902|01356801%s3567902|01356801%s35679023\n",
    "+---Sun---+---Mon---+---Tue---+---Wed---+---Thu---+---Fri---+---Sat---+\n",
    "-Sun--Mon--Tue--Wed--Thu--Fri--Sat--Sun--Mon--Tue--Wed--Thu--Fri--Sat-\n",
    "01020304050607080910111213141516171819202122232425262728293031\n",
    "01  05  09  13  17  21  25  29|01  05  09  13  17  21  25  29 \n",
    "+----------%3s----------+----------%3s----------+----------%3s----------+\n",
    "+----%3s----+----%3s----+----%3s----+----%3s----+----%3s----+----%3s----+\n",
    "+-Jan-+-Feb-+-Mar-+-Apr-+-May-+-Jun-+-Jul-+-Aug-+-Sep-+-Oct-+-Nov-+-Dec-+\n",
    "+%04d-MarAprMayJunJulAugSepOctNovDec+%04d-MarAprMayJunJulAugSepOctNovDec+\n",
    "%02dFbMrApMyJnJlAgSpOcNvDc%02dFbMrApMyJnJlAgSpOcNvDc%02dFbMrApMyJnJlAgSpOcNvDc\n",
    "%02dMAMJJASOND%02dMAMJJASOND%02dMAMJJASOND%02dMAMJJASOND%02dMAMJJASOND\n",
    "%02dMAMJJASOND%02dMAMJJASOND%02dMAMJJASOND%02dMAMJJASOND%02dMAMJJASOND%02dMAMJJASOND\n",
    "+-%04d-+-%04d-+-%04d-+-%04d-+-%04d-+-%04d-+-%04d-+-%04d-+-%04d-+-%04d-+\n" };

static int aux_inc[] = { 1, 1, 1, 1, 1, 1, 1, 1, 7, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 };

static long secs_per_period_tab[] = {
/*  HOUR   2HOUR    4HOUR    6HOUR    DAY       2DAY        3DAY */
    60*60, 60*60*2, 60*60*4, 60*60*6, 60*60*24, 60*60*24*2, 60*60*24*3,
/*  4DAY        WEEK,       2WEEK         MONTH        2MONTH       3MONTH */
    60*60*24*4, 60*60*24*7, 60*60*24*7*2, 60*60*24*31, 60*60*24*61, 60*60*24*91,
/*  6MONTH        YEAR          2YEAR           3YEAR           5YEAR */
    60*60*24*183, 60*60*24*366, 60*60*24*366*2, 60*60*24*366*3, 60*60*24*365*5,
/*  6YEAR           10YEAR */
    60*60*24*365*6, 60*60*24*365*10 };

static long secs_per_increment_tab[] = {
/*  HOUR   2HOUR  4HOUR  6HOUR  DAY       2DAY      3DAY      4DAY */
    60*60, 60*60, 60*60, 60*60, 60*60*24, 60*60*24, 60*60*24, 60*60*24,
/*  WEEK        2WEEK       MONTH        2MONTH       3MONTH       6MONTH */
    60*60*24*7, 60*60*24*7, 60*60*24*31, 60*60*24*31, 60*60*24*31, 60*60*24*31,
/*  YEAR          2YEAR         3YEAR         5YEAR */
    60*60*24*365, 60*60*24*365, 60*60*24*365, 60*60*24*365,
/*  6YEAR         10YEAR */
    60*60*24*365, 60*60*24*365 };

static char* hist_bin_name[] = {
/*  HOUR      2HOUR     4HOUR     6HOUR     DAY     2DAY    3DAY    4DAY */
    "minute", "minute", "minute", "minute", "hour", "hour", "hour", "hour",
/*  WEEK   2WEEK  MONTH  2MONTH 3MONTH  6MONTH  YEAR     2YEAR    3YEAR */
    "day", "day", "day", "day", "day",  "week", "month", "month", "year",
/*  5YEAR   6YEAR   10YEAR */
    "year", "year", "year" };

static int named_bin_size[] = {
/*  HOUR 2HOUR 4HOUR 6HOUR DAY    2DAY   3DAY   4DAY   WEEK      2WEEK */
    60,  60,   60,   60,   60*60, 60*60, 60*60, 60*60, 60*60*24, 60*60*24,
/*  MONTH     2MONTH    3MONTH    6MONTH      YEAR         2YEAR */
    60*60*24, 60*60*24, 60*60*24, 60*60*24*7, 60*60*24*31, 60*60*24*31,
/*  3YEAR        5YEAR        6YEAR        10YEAR */
    60*60*24*31, 60*60*24*31, 60*60*24*31, 60*60*24*31 };

static char* wday_name[] = { "Su", "Mo", "Tu", "We", "Th", "Fr", "Sa" };
static char* month_name[] = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

static char* argv0;


static char* timestr( time_t t );
static int find_outmode( time_t earliest_time, time_t latest_time, time_t* baseP, int* auxP );
static void usage( void ) __attribute__ ((noreturn));
static void* my_malloc( size_t size );
static void* my_realloc( void* p, size_t size );
static void out_of_mem( void ) __attribute__ ((noreturn));


int
main( int argc, char** argv )
    {
    int argn;
    int zerobase;
    int nlines;
    char* title;
    int inmode = IM_HIST, outmode, nbins;
    long secs_per_period, secs_per_bin;
    int first_line;
    char buf[10000];
    char* cp;
    time_t* raw_times = (time_t*) 0;
    float* raw_values = (float*) 0;
    int nraw, maxraw;
    time_t earliest_time, latest_time;
    int i;
    long counts[MAXBINS];
    float values[MAXBINS];
    time_t t, base, seconds;
    int aux, bin, line;
    long total_count;
    float value, total_value, average_value, hist_scale = 0.0;
    float max_value, min_value, true_max_value, true_min_value;
    float range, range_per_line, thresh;

    /* Parse arguments. */
    argv0 = argv[0];
    argn = 1;
    zerobase = 0;
    nlines = DEFAULT_LINES;
    title = (char*) 0;
    while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
	{
	if ( strcmp( argv[argn], "-z" ) == 0 )
	    zerobase = 1;
	else if ( strcmp( argv[argn], "-l" ) == 0 )
	    {
	    ++argn;
	    if ( argn >= argc )
	        usage();
	    nlines = atoi( argv[argn] );
	    }
	else if ( strcmp( argv[argn], "-t" ) == 0 )
	    {
	    ++argn;
	    if ( argn >= argc )
	        usage();
	    title = argv[argn];
	    }
	else
	    usage();
	++argn;
	}
    if ( argn != argc )
	usage();

    /* Read in the data, saving it in malloc space. */
    first_line = 1;
    maxraw = nraw = 0;
    earliest_time = 2000000000;
    latest_time = 0;
    while ( fgets( buf, sizeof(buf), stdin ) != (char*) 0 )
	{
	cp = strchr( buf, '\t' );
	if ( cp == (char*) 0 )
	    {
	    /* Histogram mode. */
	    if ( first_line )
	        {
	        inmode = IM_HIST;
		maxraw = 1000;
		raw_times = (time_t*) my_malloc( maxraw * sizeof(time_t) );
	        first_line = 0;
	        }
	    else
	        {
	        if ( inmode != IM_HIST )
		    {
		    (void) fprintf( stderr,
			"%s: mixed-mode input - both graph and histogram lines\n",
			 argv0 );
		    exit( 1 );
		    }
		if ( nraw >= maxraw )
		    {
		    maxraw *= 2;
		    raw_times = (time_t*) my_realloc(
		        (void*) raw_times, maxraw * sizeof(time_t) );
		    }
		}
	    }
	else
	    {
	    /* Graph mode. */
	    if ( first_line )
	        {
	        inmode = IM_GRAPH;
		maxraw = 1000;
		raw_times = (time_t*) my_malloc( maxraw * sizeof(time_t) );
		raw_values = (float*) my_malloc( maxraw * sizeof(float) );
	        first_line = 0;
	        }
	    else
		{
		if ( inmode != IM_GRAPH )
		    {
		    (void) fprintf( stderr,
			"%s: mixed-mode input - both graph and histogram lines\n",
			 argv0 );
		    exit( 1 );
		    }
		if ( nraw >= maxraw )
		    {
		    maxraw *= 2;
		    raw_times = (time_t*) my_realloc(
		        (void*) raw_times, maxraw * sizeof(time_t) );
		    raw_values = (float*) my_realloc(
		        (void*) raw_values, maxraw * sizeof(float) );
		    }
		}
	    *cp = '\0';
	    ++cp;
	    if ( sscanf( cp, "%g", &value ) != 1 )
		{
		(void) fprintf(
		    stderr, "%s: unparsable value - %s\n", argv0, cp );
		continue;
		}
	    raw_values[nraw] = value;
	    }

	t = date_parse( buf );
	if ( t == (time_t) -1 )
	    {
	    (void) fprintf( stderr, "%s: unparsable time - %s\n", argv0, buf );
	    continue;
	    }
	raw_times[nraw] = t;
	if ( t < earliest_time ) earliest_time = t;
	if ( t > latest_time ) latest_time = t;

	++nraw;
	}

    if ( nraw == 0 )
        {
        (void) fprintf( stderr, "%s: no data\n", argv0 );
        exit( 1 );
        }

    /* Decide on the output mode. */
    outmode = find_outmode( earliest_time, latest_time, &base, &aux );
    nbins = nbins_tab[outmode];
    secs_per_period = secs_per_period_tab[outmode];
    secs_per_bin = secs_per_period / nbins;

    /* Now loop through the saved data and stick it in bins. */
    for ( bin = 0; bin < nbins; ++bin )
        {
        counts[bin] = 0;
        values[bin] = 0.0;
        }
    total_count = 0;
    total_value = 0.0;
    true_max_value = -1.0e30;
    true_min_value = 1.0e30;
    for ( i = 0; i < nraw; ++i )
        {
	seconds = raw_times[i] - base;

	bin = seconds / secs_per_bin;
	if ( bin < 0 ) bin = 0;
	else if ( bin >= nbins ) bin = nbins - 1;
	++counts[bin];
	++total_count;
	if ( inmode == IM_GRAPH )
	    {
	    values[bin] += raw_values[i];
	    total_value += raw_values[i];
	    if ( raw_values[i] > true_max_value ) true_max_value = raw_values[i];
	    if ( raw_values[i] < true_min_value ) true_min_value = raw_values[i];
	    }
	}

    if ( zerobase )
	max_value = min_value = 0.0;
    else
	{
	max_value = -1.0e30;
	min_value = 1.0e30;
	}
    if ( inmode == IM_HIST )
	hist_scale = (float) named_bin_size[outmode] / secs_per_bin;
    for ( bin = 0; bin < nbins; ++bin )
        {
        if ( inmode == IM_GRAPH )
	    {
	    if ( counts[bin] != 0 )
		{
		values[bin] /= counts[bin];
		if ( values[bin] > max_value ) max_value = values[bin];
		if ( values[bin] < min_value ) min_value = values[bin];
		}
	    }
	else
	    {
	    values[bin] = counts[bin] * hist_scale;
	    if ( values[bin] > max_value ) max_value = values[bin];
	    if ( values[bin] < min_value ) min_value = values[bin];
	    }
	}
    if ( inmode == IM_GRAPH )
	average_value = total_value / total_count;
    else
        average_value = (float) total_count * named_bin_size[outmode] / secs_per_period;

    /* And generate the graph. */
    if ( title != (char*) 0 )
	(void) printf( "%s    ", title );
    (void) strcpy( buf, timestr( earliest_time ) );
    (void) printf( "%s through %s\n", buf, timestr( latest_time ) );
    range = max_value - min_value;
    if ( range == 0.0 ) range = nlines;
    range_per_line = range / nlines;
    for ( line = nlines - 1; line >= 0; --line )
        {
	thresh = min_value + ( (float) line + 0.5 ) * range_per_line;
	for ( bin = 0; bin < nbins; ++bin )
	    {
	    if ( counts[bin] == 0 )
	        putchar( ' ' );
	    else if ( values[bin] < thresh )
	        putchar( ' ' );
	    else
	        putchar( '*' );
	    }
	if ( thresh >= 0.001 && thresh < 10.0 )
	    (void) printf( " %.3f\n", thresh );
	else if ( thresh >= 10.0 && thresh < 100.0 )
	    (void) printf( " %.2f\n", thresh );
	else if ( thresh >= 100.0 && thresh < 1000.0 )
	    (void) printf( " %.1f\n", thresh );
	else if ( thresh >= 1000.0 && thresh < 1000000.0 )
	    (void) printf( " %.0f\n", thresh );
	else if ( thresh <= -0.001 && thresh > -10.0 )
	    (void) printf( " %.2f\n", thresh );
	else if ( thresh <= -10.0 && thresh > -100.0 )
	    (void) printf( " %.1f\n", thresh );
	else if ( thresh <= -100.0 && thresh > -100000.0 )
	    (void) printf( " %.0f\n", thresh );
	else if ( thresh < 0.0 )
	    (void) printf( " %.1g\n", thresh );
	else
	    (void) printf( " %.2g\n", thresh );
        }
    switch ( outmode )
        {
        case OM_HOUR:
	(void) printf(
	    timelabel[outmode], aux%24, aux%24, aux%24, aux%24, aux%24, aux%24,
	    aux%24, aux%24, aux%24, aux%24, aux%24, aux%24 );
        break;
        case OM_TWOHOURS:
	(void) printf(
	    timelabel[outmode], aux%24, aux%24, aux%24, aux%24, aux%24, aux%24,
	    (aux+1)%24, (aux+1)%24, (aux+1)%24,
	    (aux+1)%24, (aux+1)%24, (aux+1)%24 );
        break;
        case OM_FOURHOURS:
	(void) printf(
	    timelabel[outmode], aux%24, (aux+1)%24, (aux+2)%24, (aux+3)%24 );
        break;
        case OM_SIXHOURS:
	(void) printf(
	    timelabel[outmode], aux%24, (aux+1)%24, (aux+2)%24, (aux+3)%24,
	    (aux+4)%24, (aux+5)%24 );
        break;
        case OM_TWODAYS:
	(void) printf(
	    timelabel[outmode], wday_name[aux%7], wday_name[(aux+1)%7] );
        break;
        case OM_THREEDAYS:
	(void) printf(
	    timelabel[outmode], wday_name[aux%7], wday_name[(aux+1)%7],
	    wday_name[(aux+2)%7] );
        break;
        case OM_FOURDAYS:
	(void) printf(
	    timelabel[outmode], wday_name[aux%7], wday_name[(aux+1)%7],
	    wday_name[(aux+2)%7], wday_name[(aux+3)%7] );
        break;
        case OM_THREEMONTHS:
	(void) printf(
	    timelabel[outmode], month_name[aux%12], month_name[(aux+1)%12],
	    month_name[(aux+2)%12] );
        break;
        case OM_SIXMONTHS:
	(void) printf(
	    timelabel[outmode], month_name[aux%12], month_name[(aux+1)%12],
	    month_name[(aux+2)%12], month_name[(aux+3)%12],
	    month_name[(aux+4)%12], month_name[(aux+5)%12] );
        break;
        case OM_TWOYEARS:
	(void) printf( timelabel[outmode], aux, aux + 1 );
        break;
        case OM_THREEYEARS:
	(void) printf( timelabel[outmode], aux%100, (aux+1)%100, (aux+2)%100 );
        break;
        case OM_FIVEYEARS:
	(void) printf(
	    timelabel[outmode], aux%100, (aux+1)%100, (aux+2)%100, (aux+3)%100,
	    (aux+4)%100 );
        break;
        case OM_SIXYEARS:
	(void) printf(
	    timelabel[outmode], aux%100, (aux+1)%100, (aux+2)%100, (aux+3)%100,
	    (aux+4)%100, (aux+5)%100 );
        break;
        case OM_TENYEARS:
	(void) printf(
	    timelabel[outmode], aux, aux+1, aux+2, aux+3, aux+4, aux+5,
	    aux+6, aux+7, aux+8, aux+9 );
        break;
        default:
	(void) printf( "%s", timelabel[outmode] );
	break;
	}
    if ( inmode == IM_GRAPH )
	(void) printf(
	    "Average: %g  Maximum: %g  Minimum: %g\n",
	     average_value, true_max_value, true_min_value );
    else
        (void) printf(
            "Average rate: %g  Maximum rate: %g  Minimum rate: %g  Counts per %s\n",
            average_value, max_value, min_value, hist_bin_name[outmode] );

    exit( 0 );
    }


static char*
timestr( time_t t )
    {
    struct tm* tmP;
    static char str[20];

    tmP = localtime( &t );
    (void) strftime( str, sizeof(str), "%d%b%Y %H:%M:%S", tmP );
    return str;
    }


static int
find_outmode( time_t earliest_time, time_t latest_time, time_t* baseP, int* auxP )
    {
    struct tm* tmP;
    time_t end;
    long slop;
    int tm_sec, tm_min, tm_hour, tm_wday, tm_mday, tm_mon, tm_year, tm_isdst;
    int om;

    /* Save initial time values. */
    tmP = localtime( &earliest_time );
    tm_sec = tmP->tm_sec;
    tm_min = tmP->tm_min;
    tm_hour = tmP->tm_hour;
    tm_wday = tmP->tm_wday;
    tm_mday = tmP->tm_mday;
    tm_mon = tmP->tm_mon;
    tm_year = tmP->tm_year;
    tm_isdst = tmP->tm_isdst;

    for ( om = 0; om <= OM_LAST; ++om )
	{
	tmP->tm_sec = tm_sec;
	tmP->tm_min = tm_min;
	tmP->tm_hour = tm_hour;
	tmP->tm_wday = tm_wday;
	tmP->tm_mday = tm_mday;
	tmP->tm_mon = tm_mon;
	tmP->tm_year = tm_year;
	switch ( om )
	    {
	    case OM_TENYEARS:
	    case OM_SIXYEARS:
	    case OM_FIVEYEARS:
	    case OM_THREEYEARS:
	    case OM_TWOYEARS:
	    case OM_YEAR:
	    tmP->tm_mon = 0;
	    /* fall through */
	    case OM_SIXMONTHS:
	    case OM_THREEMONTHS:
	    case OM_TWOMONTHS:
	    case OM_MONTH:
	    tmP->tm_mday = 1;
	    /* fall through */
	    case OM_FOURDAYS:
	    case OM_THREEDAYS:
	    case OM_TWODAYS:
	    case OM_DAY:
	    tmP->tm_hour = 0;
	    /* fall through */
	    case OM_SIXHOURS:
	    case OM_FOURHOURS:
	    case OM_TWOHOURS:
	    case OM_HOUR:
	    tmP->tm_sec = 0;
	    tmP->tm_min = 0;
	    break;
	    case OM_WEEK:
	    case OM_TWOWEEKS:
	    tmP->tm_sec = 0;
	    tmP->tm_min = 0;
	    tmP->tm_hour = 0;
	    tmP->tm_mday -= tmP->tm_wday;
	    tmP->tm_wday = 0;	/* should be unnecessary */
	    break;
	    }
	switch ( om )
	    {
	    case OM_HOUR:
	    case OM_TWOHOURS:
	    case OM_FOURHOURS:
	    case OM_SIXHOURS:
	    *auxP = tm_hour;
	    break;
	    case OM_DAY:
	    case OM_TWODAYS:
	    case OM_THREEDAYS:
	    case OM_FOURDAYS:
	    *auxP = tm_wday;
	    break;
	    case OM_WEEK:
	    case OM_TWOWEEKS:
	    *auxP = tm_mday;
	    break;
	    case OM_MONTH:
	    case OM_TWOMONTHS:
	    case OM_THREEMONTHS:
	    case OM_SIXMONTHS:
	    *auxP = tm_mon;
	    break;
	    case OM_YEAR:
	    case OM_TWOYEARS:
	    case OM_THREEYEARS:
	    case OM_FIVEYEARS:
	    case OM_SIXYEARS:
	    case OM_TENYEARS:
	    *auxP = tm_year + ( tm_year > 70 ? 1900 : 2000 );
	    break;
	    }
	*baseP = mktime( tmP );
	end = *baseP + secs_per_period_tab[om];
	slop = ( end - *baseP ) * INTERVAL_SLOP;
	while ( earliest_time >= *baseP + secs_per_increment_tab[om] - slop )
	    {
	    *baseP += secs_per_increment_tab[om];
	    end += secs_per_increment_tab[om];
	    *auxP += aux_inc[om];
	    }
	if ( latest_time <= end + slop )
	    return om;
	}

    (void) fprintf( stderr, "%s: too large a time range\n", argv0 );
    exit( 1 );
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage:  %s [-z] [-l lines] [-t title]\n", argv0 );
    exit( 1 );
    }


static void*
my_malloc( size_t size )
    {
    void* r;

    r = malloc( size );
    if ( r == (void*) 0 )
	out_of_mem();
    return r;
    }


static void*
my_realloc( void* p, size_t size )
    {
    void* r;

    r = realloc( p, size );
    if ( r == (void*) 0 )
	out_of_mem();
    return r;
    }


static void
out_of_mem( void )
    {
    (void) fprintf( stderr, "%s: out of memory\n", argv0 );
    exit( 1 );
    }
