/* date_sort - sort input by date
**
** Copyright (C) 1995 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "date_parse.h"

static char* argv0;

struct line
    {
    char* text;
    time_t date;
    };

static int maxlines, nlines;
static struct line* lines;


static int line_compare( const void* v1, const void* v2 );
static void read_file( FILE* f );
static void usage( void ) __attribute__ ((noreturn));
static void* my_malloc( size_t size );
static void* my_realloc( void* p, size_t size );
static void out_of_mem( void ) __attribute__ ((noreturn));


int
main( int argc, char** argv )
    {
    int argn;
    FILE* f;
    int i;

    argv0 = argv[0];
    argn = 1;
    if ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
	usage();

    maxlines = 500;		/* arbitrary initial value */
    lines = (struct line*) my_malloc( maxlines * sizeof(struct line) );
    nlines = 0;

    if ( argn == argc )
	read_file( stdin );
    else
	while ( argn < argc )
	    {
	    if ( strcmp( argv[argn], "-" ) == 0 )
		read_file( stdin );
	    else
		{
		f = fopen( argv[argn], "r" );
		if ( f == (FILE*) 0 )
		    {
		    perror( argv[argn] );
		    exit( 1 );
		    }
		read_file( f );
		(void) fclose( f );
		}
	    ++argn;
	    }

    qsort( lines, nlines, sizeof(struct line), line_compare );

    for ( i = 0; i < nlines; ++i )
	fputs( lines[i].text, stdout );

    exit( 0 );
    }


static int
line_compare( const void* v1, const void* v2 )
    {
    const struct line* l1 = (const struct line*) v1;
    const struct line* l2 = (const struct line*) v2;

    return l1->date - l2->date;
    }


static void
read_file( FILE* f )
    {
    char aline[10000];

    /* Read until end of file. */
    while ( fgets( aline, sizeof(aline), f ) != (char*) 0 )
	{
	/* Make sure there's enough room in the lines array. */
	if ( nlines >= maxlines )
	    {
	    maxlines *= 2;
	    lines = (struct line*) my_realloc(
		(void*) lines, maxlines * sizeof(struct line) );
	    }
	lines[nlines].text = (char*) my_malloc( strlen( aline ) + 1 );
	(void) strcpy( lines[nlines].text, aline );
	lines[nlines].date = date_parse( aline );
	if ( lines[nlines].date == (time_t) -1 )
	    (void) fprintf( stderr, "%s: unparsable date - %s", argv0, aline );
	++nlines;
	}
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage:  %s [file ...]\n", argv0 );
    exit( 1 );
    }


static void*
my_malloc( size_t size )
    {
    void* r;

    r = malloc( size );
    if ( r == (void*) 0 )
	out_of_mem();
    return r;
    }


static void*
my_realloc( void* p, size_t size )
    {
    void* r;

    r = realloc( p, size );
    if ( r == (void*) 0 )
	out_of_mem();
    return r;
    }


static void
out_of_mem( void )
    {
    (void) fprintf( stderr, "%s: out of memory\n", argv0 );
    exit( 1 );
    }
