/* date_merge - merge files by date
**
** Copyright (C) 2002 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "date_parse.h"

static char* argv0;


static void usage( void ) __attribute__ ((noreturn));
static void* my_realloc( void* p, size_t size );
static void* my_malloc( size_t size );
static void out_of_mem( void ) __attribute__ ((noreturn));


int
main( int argc, char** argv )
    {
    int nfiles;
    FILE** ifps;
    int* eofs;
    int* needlines;
    char** lines;
    size_t* linesizes;
    time_t* dates;
    int i, neofs;
    char aline[10000];
    int earliest_idx;
    time_t earliest_date;
    size_t len;

    /* Initialize. */
    argv0 = argv[0];
    if ( argc < 3 )
	usage();
    nfiles = argc - 1;
    ifps = (FILE**) my_malloc( nfiles * sizeof(FILE*) );
    eofs = (int*) my_malloc( nfiles * sizeof(int) );
    needlines = (int*) my_malloc( nfiles * sizeof(int) );
    lines = (char**) my_malloc( nfiles * sizeof(char*) );
    linesizes = (size_t*) my_malloc( nfiles * sizeof(size_t) );
    dates = (time_t*) my_malloc( nfiles * sizeof(time_t) );

    /* Open and prime files. */
    for ( i = 0; i < nfiles; ++i )
	{
	ifps[i] = fopen( argv[i+1], "r" );
	if ( ifps[i] == (FILE*) 0 )
	    {
	    perror( argv[i+1] );
	    exit( 1 );
	    }
	needlines[i] = 1;
	eofs[i] = 0;
	linesizes[i] = 200;	/* arbitrary initial allocation */
	lines[i] = (char*) my_malloc( linesizes[i] );
	}

    /* Loop until all files are EOF. */
    neofs = 0;
    for (;;)
	{

	/* Read in a line from any file that needs it.  The first time
	** through this will be all the files; after that, only one of them.
	*/
	for ( i = 0; i < nfiles; ++i )
	    if ( ( ! eofs[i] ) && needlines[i] )
		{
		if ( fgets( aline, sizeof(aline), ifps[i] ) == (char*) 0 )
		    {
		    eofs[i] = 1;
		    ++neofs;
		    }
		else
		    {
		    needlines[i] = 0;
		    len = strlen( aline );
		    if ( len + 1 > linesizes[i] )
			{
			linesizes[i] = ( len + 1 ) * 2;
			lines[i] = my_realloc( (void*) lines[i], linesizes[i] );
			}
		    (void) strcpy( lines[i], aline );
		    dates[i] = date_parse( lines[i] );
		    if ( dates[i] == (time_t) -1 )
			(void) fprintf( stderr, "%s: unparsable date - %s\n", argv0, lines[i] );
			/* (But keep the line.) */
		    }
		}

	/* Are we done? */
	if ( neofs == nfiles )
	    break;

	/* Find the line with the earliest date and write it out.
	** Since parse failures have a date of -1, they get written
	** out as soon as they are read.
	*/
	earliest_idx = -1;
	earliest_date = 2147483647L;
	for ( i = 0; i < nfiles; ++i )
	    {
	    if ( ( ! eofs[i] ) && dates[i] < earliest_date )
		{
		earliest_idx = i;
		earliest_date = dates[i];
		}
	    }
	if ( earliest_idx == -1 )
	    {
	    (void) fprintf( stderr, "%s: shouldn't happen!\n", argv0 );
	    exit( 1 );
	    }
	fputs( lines[earliest_idx], stdout );
	needlines[earliest_idx] = 1;

	}

    /* Done. */
    for ( i = 0; i < nfiles; ++i )
	(void) fclose( ifps[i] );
    free( (void*) ifps );
    free( (void*) eofs );
    free( (void*) needlines );
    free( (void*) lines );
    free( (void*) linesizes );
    free( (void*) dates );
    exit( 0 );
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage:  %s file1 file2  ...\n", argv0 );
    exit( 1 );
    }


static void*
my_realloc( void* p, size_t size )
    {
    void* r;

    r = realloc( p, size );
    if ( r == (void*) 0 )
	out_of_mem();
    return r;
    }


static void*
my_malloc( size_t size )
    {
    void* r;

    r = malloc( size );
    if ( r == (void*) 0 )
	out_of_mem();
    return r;
    }


static void
out_of_mem( void )
    {
    (void) fprintf( stderr, "%s: out of memory\n", argv0 );
    exit( 1 );
    }
