/* ll2utm.c - convert latitude & longitude into Universal Transverse Mercator
**
** Partially based on code by Chuck Gantz <chuck.gantz@globalstar.com>.
**
** Copyright  2001 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>

#include "coords.h"


static char* argv0;


static void usage( void );
static void parse_error( void );
static int nsmul( char* nsstr );
static int ewmul( char* ewstr );
static char utm_letter( double latitude );


int
main( int argc, char** argv )
    {
    int arglen, argn;
    char* argstr;
    char* cp;
    double latitude, lat_deg, lat_min, lat_sec;
    double longitude, long_deg, long_min, long_sec;
    char nsstr[100], ewstr[100];
    double lat_rad, long_rad;
    double long_origin, long_origin_rad;
    double eccPrimeSquared;
    double N, T, C, A, M;
    int zone;
    double northing, easting;

    argv0 = argv[0];
    if ( argc < 2 )
	usage();

    /* Collect all args into a single string. */
    arglen = 0;
    for ( argn = 1; argn < argc; ++argn )
	arglen += strlen( argv[argn] + 1 );
    argstr = (char*) malloc( arglen );
    if ( argstr == (char*) 0 )
	{
	(void) fprintf( stderr, "%s: out of memory\n", argv0 );
	exit( 1 );
	}
    (void) strcpy( argstr, "" );
    for ( argn = 1; argn < argc; ++argn )
	{
	if ( argn > 1 )
	    (void) strcat( argstr, " " );
	(void) strcat( argstr, argv[argn] );
	}

    /* Turn punctuation into blanks. */
    for ( cp = argstr; *cp != '\0'; ++cp )
	/* Handle %-escapes specially. */
	if ( *cp == '%' )
	    {
	    *cp = ' ';
	    if ( *(cp+1) != '\0' )
	      *(++cp) = ' ';
	    if ( *(cp+1) != '\0' )
	      *(++cp) = ' ';
	    }
	/* Now blank anything that's not a valid lat-long char. */
	else if ( strchr( "0123456789.-NnSsEeWw", *cp ) == (char*) 0 )
	    *cp = ' ';

    /* Trim leading and trailing blanks. */
    while ( argstr[0] == ' ' )
	(void) strcpy( argstr, &(argstr[1]) );
    while ( argstr[strlen(argstr)-1] == ' ' )
	argstr[strlen(argstr)-1] = '\0';

    /* Parse string into latitude and longitude. */
    if ( sscanf( argstr, "%lf-%lf-%lf%[nsNS] / %lf-%lf-%lf%[ewEW]", &lat_deg, &lat_min, &lat_sec, nsstr, &long_deg, &long_min, &long_sec, ewstr ) == 8 )
	{
	latitude = lat_deg + lat_min / 60.0 + lat_sec / 60.0 / 60.0;
	longitude = long_deg + long_min / 60.0 + long_sec / 60.0 / 60.0;
	latitude *= nsmul( nsstr );
	longitude *= ewmul( ewstr );
	}
    else if ( sscanf( argstr, "%[nsNS] %lf %lf %lf %[ewEW] %lf %lf %lf", nsstr, &lat_deg, &lat_min, &lat_sec, ewstr, &long_deg, &long_min, &long_sec ) == 8 ||
              sscanf( argstr, "%lf %lf %lf %[nsNS] %lf %lf %lf %[ewEW]", &lat_deg, &lat_min, &lat_sec, nsstr, &long_deg, &long_min, &long_sec, ewstr ) == 8 )
	{
	latitude = lat_deg + lat_min / 60.0 + lat_sec / 60.0 / 60.0;
	longitude = long_deg + long_min / 60.0 + long_sec / 60.0 / 60.0;
	latitude *= nsmul( nsstr );
	longitude *= ewmul( ewstr );
	}
    else if ( sscanf( argstr, "%lf %lf %lf %lf %lf %lf", &lat_deg, &lat_min, &lat_sec, &long_deg, &long_min, &long_sec ) == 6 )
	{
	latitude = lat_deg + lat_min / 60.0 + lat_sec / 60.0 / 60.0;
	longitude = long_deg + long_min / 60.0 + long_sec / 60.0 / 60.0;
	}
    else if ( sscanf( argstr, "%[nsNS] %lf %lf %[ewEW] %lf %lf", nsstr, &lat_deg, &lat_min, ewstr, &long_deg, &long_min ) == 6 ||
              sscanf( argstr, "%lf %lf %[nsNS] %lf %lf %[ewEW]", &lat_deg, &lat_min, nsstr, &long_deg, &long_min, ewstr ) == 6 )
	{
	latitude = lat_deg + lat_min / 60.0;
	longitude = long_deg + long_min / 60.0;
	latitude *= nsmul( nsstr );
	longitude *= ewmul( ewstr );
	}
    else if ( sscanf( argstr, "%lf %lf %lf %lf", &lat_deg, &lat_min, &long_deg, &long_min ) == 4 )
	{
	latitude = lat_deg + lat_min / 60.0;
	longitude = long_deg + long_min / 60.0;
	}
    else if ( sscanf( argstr, "%lf-%lf x %lf-%lf", &lat_deg, &lat_min, &long_deg, &long_min ) == 4 )
	{
	latitude = lat_deg + lat_min / 60.0;
	longitude = long_deg + long_min / 60.0;
	longitude *= -1;
	}
    else if ( sscanf( argstr, "%[nsNS] %lf %[ewEW] %lf", nsstr, &latitude, ewstr, &longitude ) == 4 ||
              sscanf( argstr, "%lf %[nsNS] %lf %[ewEW]", &latitude, nsstr, &longitude, ewstr ) == 4 )
	{
	latitude *= nsmul( nsstr );
	longitude *= ewmul( ewstr );
	}
    else if ( sscanf( argstr, "%lf %lf", &latitude, &longitude ) == 2 )
	;
    else
	parse_error();

    /* We want the longitude within -180..180. */
    if ( longitude < -180.0 )
	longitude += 360.0;
    if ( longitude > 180.0 )
	longitude -= 360.0;

    /* Now convert. */
    lat_rad = latitude * M_PI / 180.0;
    long_rad = longitude * M_PI / 180.0;
    zone = (int) ( ( longitude + 180 ) / 6 ) + 1;
    if ( latitude >= 56.0 && latitude < 64.0 &&
	 longitude >= 3.0 && longitude < 12.0 )
	zone = 32;
    /* Special zones for Svalbard. */
    if ( latitude >= 72.0 && latitude < 84.0 )
	{
	if      ( longitude >= 0.0  && longitude <  9.0 ) zone = 31;
	else if ( longitude >= 9.0  && longitude < 21.0 ) zone = 33;
	else if ( longitude >= 21.0 && longitude < 33.0 ) zone = 35;
	else if ( longitude >= 33.0 && longitude < 42.0 ) zone = 37;
	}
    long_origin = ( zone - 1 ) * 6 - 180 + 3;	/* +3 puts origin in middle of zone */
    long_origin_rad = long_origin * M_PI / 180.0;
    eccPrimeSquared = EccentricitySquared / ( 1.0 - EccentricitySquared );
    N = EquatorialRadius / sqrt( 1.0 - EccentricitySquared * sin( lat_rad ) * sin( lat_rad ) );
    T = tan( lat_rad ) * tan( lat_rad );
    C = eccPrimeSquared * cos( lat_rad ) * cos( lat_rad );
    A = cos( lat_rad ) * ( long_rad - long_origin_rad );
    M = EquatorialRadius * ( ( 1.0 - EccentricitySquared / 4 - 3 * EccentricitySquared * EccentricitySquared / 64 - 5 * EccentricitySquared * EccentricitySquared * EccentricitySquared / 256 ) * lat_rad - ( 3 * EccentricitySquared / 8 + 3 * EccentricitySquared * EccentricitySquared / 32 + 45 * EccentricitySquared * EccentricitySquared * EccentricitySquared / 1024 ) * sin( 2 * lat_rad ) + ( 15 * EccentricitySquared * EccentricitySquared / 256 + 45 * EccentricitySquared * EccentricitySquared * EccentricitySquared / 1024 ) * sin( 4 * lat_rad ) - ( 35 * EccentricitySquared * EccentricitySquared * EccentricitySquared / 3072 ) * sin( 6 * lat_rad ) );
    easting =
	K0 * N * ( A + ( 1 - T + C ) * A * A * A / 6 + ( 5 - 18 * T + T * T + 72 * C - 58 * eccPrimeSquared ) * A * A * A * A * A / 120 ) + 500000.0;
    northing =
	K0 * ( M + N * tan( lat_rad ) * ( A * A / 2 + ( 5 - T + 9 * C + 4 * C * C ) * A * A * A * A / 24 + ( 61 - 58 * T + T * T + 600 * C - 330 * eccPrimeSquared ) * A * A * A * A * A * A / 720 ) );
    if ( latitude < 0.0 )
	northing += 10000000.0;  /* 1e7 meter offset for southern hemisphere */

    /* Show results. */
    (void) printf(
	"%ld %ld %d%c\n", (long) northing, (long) easting,
	zone, utm_letter( latitude ) );

    /* All done. */
    exit( 0 );
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage:  %s latitude longitude\n", argv0 );
    exit( 1 );
    }


static void
parse_error( void )
    {
    (void) fprintf( stderr, "%s: can't parse latitude and longitude\n", argv0 );
    exit( 1 );
    }


static int
nsmul( char* nsstr )
    {
    switch ( nsstr[0] )
	{
	case 'n': case 'N':
	return 1;
	case 's': case 'S':
	return -1;
	default:
	parse_error();
	return 0;
	}
    }


static int
ewmul( char* ewstr )
    {
    switch ( ewstr[0] )
	{
	case 'e': case 'E':
	return 1;
	case 'w': case 'W':
	return -1;
	default:
	parse_error();
	return 0;
	}
    }

static char
utm_letter( double latitude )
    {
    /* This routine determines the correct UTM letter designator for the
    ** given latitude.  It returns 'Z' if the latitude is outside the UTM
    ** limits of 84N to 80S.
    */
    if ( latitude <= 84.0 && latitude >= 72.0 ) return 'X';
    else if ( latitude < 72.0 && latitude >= 64.0 ) return 'W';
    else if ( latitude < 64.0 && latitude >= 56.0 ) return 'V';
    else if ( latitude < 56.0 && latitude >= 48.0 ) return 'U';
    else if ( latitude < 48.0 && latitude >= 40.0 ) return 'T';
    else if ( latitude < 40.0 && latitude >= 32.0 ) return 'S';
    else if ( latitude < 32.0 && latitude >= 24.0 ) return 'R';
    else if ( latitude < 24.0 && latitude >= 16.0 ) return 'Q';
    else if ( latitude < 16.0 && latitude >= 8.0 ) return 'P';
    else if ( latitude <  8.0 && latitude >= 0.0 ) return 'N';
    else if ( latitude <  0.0 && latitude >= -8.0 ) return 'M';
    else if ( latitude < -8.0 && latitude >= -16.0 ) return 'L';
    else if ( latitude < -16.0 && latitude >= -24.0 ) return 'K';
    else if ( latitude < -24.0 && latitude >= -32.0 ) return 'J';
    else if ( latitude < -32.0 && latitude >= -40.0 ) return 'H';
    else if ( latitude < -40.0 && latitude >= -48.0 ) return 'G';
    else if ( latitude < -48.0 && latitude >= -56.0 ) return 'F';
    else if ( latitude < -56.0 && latitude >= -64.0 ) return 'E';
    else if ( latitude < -64.0 && latitude >= -72.0 ) return 'D';
    else if ( latitude < -72.0 && latitude >= -80.0 ) return 'C';
    else return 'Z';
    }
