/* closest_fractions - show the closest rational approximations to a real
**
** Copyright  2000 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MAX_TERM 100
#define NEARBY 5

#define min(a,b) ((a)<(b)?(a):(b))

/* Struct to hold candidate fractions. */
struct frac {
    int num, denom;
    double val, diff;
    };

static char* argv0;


static void
usage( void )
    {
    (void) fprintf( stderr, "usage: %s [-m max_term] [-n nearby] val ...\n", argv0 );
    exit( 1 );
    }


/* Euclid's algorithm. */
static int
gcd( int a, int b )
    {
    int m;

    for (;;)
	{
	if ( b == 0 )
	    return a;
	m = a % b;
	a = b;
	b = m;
	}
    }

/* Determine if two numbers are relatively prime. */
static int
relprime( int a, int b )
    {
    return gcd( a, b ) == 1;
    }


/* Comparison routines for qsort. */

static int
compar( const void* p1, const void* p2 )
    {
    const struct frac* f1 = (struct frac*) p1;
    const struct frac* f2 = (struct frac*) p2;
    double d;

    d = f1->val - f2->val;
    if ( d < 0.0 )
	return -1;
    else if ( d > 0.0 )
	return 1;
    else
	return 0;
    }

static int
compar_rev( const void* p1, const void* p2 )
    {
    const struct frac* f1 = (struct frac*) p1;
    const struct frac* f2 = (struct frac*) p2;
    double d;

    d = f1->val - f2->val;
    if ( d < 0.0 )
	return 1;
    else if ( d > 0.0 )
	return -1;
    else
	return 0;
    }


int
main( int argc, char** argv )
    {
    int max_term, nearby;
    int argn;
    struct frac* lower;
    struct frac* higher;
    int nlower, nhigher;
    int num, denom, num2, i;
    double val, val2, diff;

    argv0 = argv[0];
    max_term = MAX_TERM;
    nearby = NEARBY;
    argn = 1;
    while ( argn < argc && argv[argn][0] == '-' )
	{
	if ( strcmp( argv[argn], "-m" ) == 0 )
	    {
	    ++argn;
	    max_term = atoi( argv[argn] );
	    }
	else if ( strcmp( argv[argn], "-n" ) == 0 )
	    {
	    ++argn;
	    nearby = atoi( argv[argn] );
	    }
	else
	    usage();
	++argn;
	}
    if ( argn == argc )
	usage();

    higher = (struct frac*) malloc( ( nearby + 1 ) * sizeof(struct frac) );
    lower = (struct frac*) malloc( ( nearby + 1 ) * sizeof(struct frac) );
    if ( higher == (struct frac*) 0 || lower == (struct frac*) 0 )
	{
	(void) fprintf( stderr, "%s: out of memory\n", argv0 );
	exit( 1 );
	}

    for ( ; argn < argc; ++argn )
        {
	val = atof( argv[argn] );
	nhigher = nlower = 0;

	/* Loop through all possible denominators. */
	for ( denom = 1; denom <= max_term; ++denom )
	    {
	    /* Figure out the closest numerator. */
	    num = val * denom + 0.5;
	    /* Loop through all nearby numerators. */
	    for ( num2 = num - nearby; num2 <= num + nearby; ++num2 )
		{
		/* Make sure it's in range. */
		if ( num2 < 1 || num2 > max_term )
		    continue;
		/* If they're not relatively prime then the fraction
		** isn't in lowest terms, so skip it.
		*/
		if ( ! relprime( num2, denom ) )
		    continue;
		/* Ok.  Add this fraction to either the higher or lower
		** table, and sort to eliminate the worst.
		*/
		val2 = (double) num2 / (double) denom;
		diff = val2 - val;
		if ( diff < 0.0 )
		    {
		    lower[nlower].num = num2;
		    lower[nlower].denom = denom;
		    lower[nlower].val = val2;
		    lower[nlower].diff = diff;
		    if ( nlower > 0 && diff > lower[nlower - 1].diff )
			qsort( (void*) lower, nlower + 1, sizeof(struct frac), compar_rev );
		    nlower = min( nlower + 1, nearby );
		    }
		else
		    {
		    higher[nhigher].num = num2;
		    higher[nhigher].denom = denom;
		    higher[nhigher].val = val2;
		    higher[nhigher].diff = diff;
		    if ( nhigher > 0 && diff < higher[nhigher - 1].diff )
			qsort( (void*) higher, nhigher + 1, sizeof(struct frac), compar );
		    nhigher = min( nhigher + 1, nearby );
		    }
		}
	    }

	/* Show the nearby fractions. */
	(void) printf( "%.16g:\n", val );
	for ( i = nearby - 1; i >= 0; --i )
	    (void) printf( "    %3d / %3d  =  %-17.16g  (%.6g)\n", lower[i].num, lower[i].denom, lower[i].val, lower[i].diff );
	for ( i = 0; i < nearby; ++i )
	    (void) printf( "    %3d / %3d  =  %-17.16g  (%.6g)\n", higher[i].num, higher[i].denom, higher[i].val, higher[i].diff );
	}

    exit( 0 );
    }
