/* blackmilter - blacklist mail filter module
**
** Copyright  2004 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/

#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_GRP_H
#include <grp.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#include "libmilter/mfapi.h"

#include "version.h"
#include "iparray.h"


/* Defines. */

#define HEADER "X-IP-Blacklisted"

#define MAX_LISTS 100		/* max number of blacklist or whitelist files */
#define MIN_UPDATE_INTERVAL 10	/* min seconds between updates */
#define MIN_FILE_AGE 30		/* min age of changed file before reading */

#define max(a,b) ((a) > (b) ? (a) : (b))


/* Forwards. */

static void usage( void );
static int check_files( void );
static int check_file( char* filename );
static int stat_files( time_t current_time );
static int stat_file( time_t current_time, time_t mtime, char* filename );
static void read_files( void );
static time_t read_file( iparray list, char* filename );
static void handle_sigusr1( int sig );
static void update( void );
static void trim( char* str );
static sfsistat black_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr );
static sfsistat black_helo( SMFICTX* ctx, char* helohost );
static sfsistat black_header( SMFICTX* ctx, char* name, char* value );
static sfsistat black_eom( SMFICTX* ctx );
static sfsistat black_close( SMFICTX* ctx );


/* Globals. */

static char* argv0;
static int got_usr1;
static time_t last_update;
static pthread_mutex_t lock;

static char* blacklist_files[MAX_LISTS];
static time_t blacklist_mtimes[MAX_LISTS];
static char* whitelist_files[MAX_LISTS];
static time_t whitelist_mtimes[MAX_LISTS];
static int n_blacklist_files, n_blacklists, n_whitelist_files, n_whitelists;
static int autoupdate, markonly, graylist, loglistname, nodaemon;
static int threshold;
static char* updatesocket;
static iparray blacklists[MAX_LISTS];
static iparray whitelists[MAX_LISTS];
static char* rejectmessage;
static char rejectmessage_str[1000];
static char* user;

static struct smfiDesc smfilter =
    {
    "BLACK",		/* filter name */
    SMFI_VERSION,	/* version code -- do not change */
    0,			/* flags */
    black_connect,	/* connection info filter */
    black_helo,		/* SMTP HELO command filter */
    NULL,		/* envelope sender filter */
    NULL,		/* envelope recipient filter */
    black_header,	/* header filter */
    NULL,		/* end of header */
    NULL,		/* body block filter */
    black_eom,		/* end of message */
    NULL,		/* message aborted */
    black_close		/* connection cleanup */
    };


int
main( int argc, char** argv )
    {
    int argn;
    char* sockarg;
    int i;

    argv0 = strrchr( argv[0], '/' );
    if ( argv0 != (char*) 0 )
	++argv0;
    else
	argv0 = argv[0];

    /* Parse args. */
    n_blacklist_files = 0;
    n_whitelist_files = 0;
    rejectmessage = (char*) 0;
    autoupdate = 0;
    threshold = 1;
    updatesocket = (char*) 0;
    markonly = 0;
    graylist = 0;
    loglistname = 0;
    user = (char*) 0;
    nodaemon = 0;
    argn = 1;
    while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
	{
	if ( strncmp( argv[argn], "-blacklist", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    if ( n_blacklist_files == MAX_LISTS )
		{
		(void) fprintf( stderr, "%s: too many blacklist files\n", argv0 );
		exit( 1 );
		}
	    ++argn;
	    blacklist_files[n_blacklist_files++] = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-whitelist", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    if ( n_whitelist_files == MAX_LISTS )
		{
		(void) fprintf( stderr, "%s: too many whitelist files\n", argv0 );
		exit( 1 );
		}
	    ++argn;
	    whitelist_files[n_whitelist_files++] = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-rejectmessage", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    rejectmessage = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-autoupdate", strlen( argv[argn] ) ) == 0 )
	    autoupdate = 1;
	else if ( strncmp( argv[argn], "-threshold", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    threshold = atoi( argv[argn] );
	    }
	else if ( strncmp( argv[argn], "-updatesocket", max( strlen( argv[argn] ), 3 ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    updatesocket = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-markonly", strlen( argv[argn] ) ) == 0 )
	    markonly = 1;
	else if ( strncmp( argv[argn], "-graylist", strlen( argv[argn] ) ) == 0 ||
	          strncmp( argv[argn], "-greylist", strlen( argv[argn] ) ) == 0 )
	    graylist = 1;
	else if ( strncmp( argv[argn], "-loglistname", strlen( argv[argn] ) ) == 0 )
	    loglistname = 1;
	else if ( strncmp( argv[argn], "-user", max( strlen( argv[argn] ), 3 ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    user = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-nodaemon", strlen( argv[argn] ) ) == 0 )
	    nodaemon = 1;
	else if ( strncmp( argv[argn], "-X", strlen( argv[argn] ) ) == 0 )
	    nodaemon = 1;
	else
	    usage();
	++argn;
	}
    if ( argn >= argc )
	usage();
    sockarg = argv[argn++];
    if ( argn != argc )
	usage();
    if ( n_blacklist_files < 1 || threshold < 1 )
	usage();
    if ( markonly && graylist )
	{
	(void) fprintf( stderr, "%s: -markonly and -graylist are mutually exclusive\n", argv0 );
	exit( 1 );
	}

    if ( getuid() == 0 )
	{
	/* If we're root, the very first thing we do is become another user. */
	if ( user == (char*) 0 )
	    (void) fprintf( stderr, "%s: warning - started as root but no -user flag specified\n", argv0 );
	else
	    {
	    struct passwd* pwd = getpwnam( user );
	    if ( pwd == (struct passwd*) 0 )
		{
		(void) fprintf( stderr, "%s: unknown user - '%s'\n", argv0, user );
		exit( 1 );
		}
	    /* Set aux groups to null. */
	    if ( setgroups( 0, (gid_t*) 0 ) < 0 )
		{
		perror( "setgroups" );
		exit( 1 );
		}
	    /* Set primary group. */
	    if ( setgid( pwd->pw_gid ) < 0 )
		{
		perror( "setgid" );
		exit( 1 );
		}
	    /* Try setting aux groups correctly - not critical if this fails. */
	    if ( initgroups( user, pwd->pw_gid ) < 0 )
		perror( "initgroups" );
	    /* Set uid. */
	    if ( setuid( pwd->pw_uid ) < 0 )
		{
		perror( "setuid" );
		exit( 1 );
		}
	    }
	}
    else
	{
	/* If we're not root but got a -user flag anyway, that's an error. */
	if ( user != (char*) 0 )
	    (void) fprintf( stderr, "%s: can't switch users if not started as root\n", argv0 );
	    exit( 1 );
	}

    openlog( argv0, 0, LOG_MAIL );

    /* Initialize iparrays. */
    if ( loglistname )
	{
	n_blacklists = n_blacklist_files;
	n_whitelists = n_whitelist_files;
	}
    else
	{
	n_blacklists = 1;
	if ( n_whitelist_files > 0 )
	    n_whitelists = 1;
	else
	    n_whitelists = 0;
	}
    for ( i = 0; i < n_blacklists; ++i )
	{
	blacklists[i] = iparray_new();
	if ( blacklists[i] == (iparray) 0 )
	    {
	    (void) fprintf( stderr, "%s: blacklist create failed\n", argv0 );
	    exit( 1 );
	    }
	}
    for ( i = 0; i < n_whitelists; ++i )
	{
	whitelists[i] = iparray_new();
	if ( whitelists[i] == (iparray) 0 )
	    {
	    (void) fprintf( stderr, "%s: whitelist create failed\n", argv0 );
	    exit( 1 );
	    }
	}
    if ( ! check_files() )
	exit( 1 );
    if ( rejectmessage == (char*) 0 )
	{
	(void) snprintf( rejectmessage_str, sizeof(rejectmessage_str), "IP address blocked by %s %s - %s", BLACKMILTER_PROGRAM, BLACKMILTER_VERSION, BLACKMILTER_URL );
	rejectmessage = rejectmessage_str;
	}

    if ( pthread_mutex_init( &lock, (pthread_mutexattr_t*) 0 ) != 0 )
	{
	(void) fprintf( stderr, "%s: pthread_mutex_init failed\n", argv0 );
	exit( 1 );
	}

    /* Remove old local socket.  Actually we should probably stat the file
    ** and check that it is in fact a socket before removing it.
    */
    if ( strncasecmp( sockarg, "unix:", 5 ) == 0 )
	{
	if ( unlink( &sockarg[5] ) < 0 )
	    if ( errno != ENOENT )
		perror( &sockarg[5] );
	}
    else if( strncasecmp( sockarg, "local:", 6 ) == 0 )
	{
	if ( unlink( &sockarg[6] ) < 0 )
	    if ( errno != ENOENT )
		perror( &sockarg[6] );
	}
    else if ( strchr( sockarg, ':' ) == (char*) 0 )
	{
	if ( unlink( sockarg ) < 0 )
	    if ( errno != ENOENT )
		perror( sockarg );
	}

    /* Harden our umask so that the new socket gets created securely. */
    umask( 0077 );

    (void) smfi_setconn( sockarg );
    if ( markonly )
	smfilter.xxfi_flags |= SMFIF_CHGHDRS|SMFIF_ADDHDRS;
    if ( smfi_register( smfilter ) == MI_FAILURE )
	{
	(void) fprintf( stderr, "%s: register failed\n", argv0 );
	exit( 1 );
	}

    if ( ! nodaemon )
	{
	/* Daemonize. */
#ifdef HAVE_DAEMON
	if ( daemon( 0, 0 ) < 0)
	    {
	    perror( "daemon" );
	    exit( 1 );
	    }
#else /* HAVE_DAEMON */
	switch ( fork() )
	    {
	    case 0:
		break;  
	    case -1:
		syslog( LOG_CRIT, "fork: %m" );
		perror( "fork" );
		exit( 1 );
	    default:
		exit( 0 );
	    }
#ifdef HAVE_SETSID
	setsid();
#endif /* HAVE_SETSID */
#endif /* HAVE_DAEMON */
	}

    read_files();
    last_update = time( (time_t*) 0 );
    got_usr1 = 0;
    (void) signal( SIGUSR1, handle_sigusr1 );

    if ( smfi_main() == MI_FAILURE )
	{
	syslog( LOG_ERR, "smfi_main() failed" );
	exit( 1 );
	}

    (void) pthread_mutex_destroy( &lock );
    for ( i = 0; i < n_blacklists; ++i )
	iparray_delete( blacklists[i] );
    for ( i = 0; i < n_whitelists; ++i )
	iparray_delete( whitelists[i] );
    iparray_fini();
    closelog();
    exit( 0 );
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage:  %s [-blacklist file] [-whitelist file] [-rejectmessage msg] [-autoupdate] [-threshold N] [-updatesocket socket] [-markonly] [-graylist] [-loglistname] [-user user] [-nodaemon|-X] socket\n", argv0 );
    exit( 1 );
    }


/* Returns 1 if the files are all readable, else 0. */
static int
check_files( void )
    {
    int i;

    for ( i = 0; i < n_blacklist_files; ++i )
	if ( ! check_file( blacklist_files[i] ) )
	    return 0;
    for ( i = 0; i < n_whitelist_files; ++i )
	if ( ! check_file( whitelist_files[i] ) )
	    return 0;
    return 1;
    }


/* Returns 1 if the file is readable, else 0. */
static int
check_file( char* filename )
    {
    FILE* fp;

    fp = fopen( filename, "r" );
    if ( fp == (FILE*) 0 )
	{
	perror( filename );
	return 0;
	}
    (void) fclose( fp );
    return 1;
    }


/* Returns 1 if all files are still current, else 0. */
static int
stat_files( time_t current_time )
    {
    int i;

    for ( i = 0; i < n_blacklist_files; ++i )
	if ( ! stat_file( current_time, blacklist_mtimes[i], blacklist_files[i] ) )
	    return 0;
    for ( i = 0; i < n_whitelist_files; ++i )
	if ( ! stat_file( current_time, whitelist_mtimes[i], whitelist_files[i] ) )
	    return 0;
    return 1;
    }


/* Returns 1 if the file is still current, else 0. */
static int
stat_file( time_t current_time, time_t mtime, char* filename )
    {
    struct stat sb;

    if ( stat( filename, &sb ) < 0 )
	return 1;	/* Can't stat it?  Ignore. */
    if ( sb.st_mtime == mtime )
	return 1;	/* Unchanged. */
    if ( current_time - sb.st_mtime < MIN_FILE_AGE )
	return 1;	/* Not old enough, we'll catch it next time. */
    return 0;		/* Changed. */
    }


/* Reads all the files into the database. */
static void
read_files( void )
    {
    int i;

    for ( i = 0; i < n_blacklist_files; ++i )
	blacklist_mtimes[i] = read_file( loglistname ? blacklists[i] : blacklists[0], blacklist_files[i] );
    for ( i = 0; i < n_whitelist_files; ++i )
	whitelist_mtimes[i] = read_file( loglistname ? whitelists[i] : whitelists[0], whitelist_files[i] );
    }


/* Reads one file into the database. */
static time_t
read_file( iparray list, char* filename )
    {
    FILE* fp;
    struct stat sb;
    time_t mtime;
    char line[10000];
    octets ip;

    syslog( LOG_INFO, "reading %s", filename );
    fp = fopen( filename, "r" );
    if ( fp == (FILE*) 0 )
	{
	syslog( LOG_ERR, "%s - %m", filename );
	mtime = (time_t) -1;
	}
    else
	{
	if ( fstat( fileno(fp), &sb ) == 0 )
	    mtime = sb.st_mtime;
	else
	    mtime = (time_t) -1;
	while ( fgets( line, sizeof(line), fp ) != (char*) 0 )
	    {
	    trim( line );
	    if ( line[0] == '\0' )
		continue;
	    if ( iparray_parse_octets( line, &ip ) )
		(void) iparray_incr( list, ip );
	    else
		syslog( LOG_INFO, "unparsable IP address - \"%s\"", line );
	    }
	(void) fclose( fp );
	}
    return mtime;
    }


/* SIGUSR1 says to re-open the data files. */
static void
handle_sigusr1( int sig )
    {
    const int oerrno = errno;
		
    /* Set up handler again. */
    (void) signal( SIGUSR1, handle_sigusr1 );

    /* Just set a flag that we got the signal. */
    got_usr1 = 1;

    /* Restore previous errno. */
    errno = oerrno;
    }


static void
update( void )
    {
    time_t current_time;
    int i;

    current_time = time( (time_t*) 0 );
    if ( current_time - last_update < MIN_UPDATE_INTERVAL )
	return;
    last_update = current_time;

    if ( pthread_mutex_lock( &lock ) == 0 )
	{
	if ( got_usr1 )
	    {
	    syslog( LOG_INFO, "received SIGUSR1 - updating database" );
	    for ( i = 0; i < n_blacklists; ++i )
		iparray_clear( blacklists[i] );
	    for ( i = 0; i < n_whitelists; ++i )
		iparray_clear( blacklists[i] );
	    read_files();
	    got_usr1 = 0;
	    }
	else if ( autoupdate )
	    {
	    if ( ! stat_files( current_time ) )
		{
		syslog( LOG_INFO, "database files changed - autoupdating" );
		for ( i = 0; i < n_blacklists; ++i )
		    iparray_clear( blacklists[i] );
		for ( i = 0; i < n_whitelists; ++i )
		    iparray_clear( blacklists[i] );
		read_files();
		}
	    }

	if ( updatesocket != (char*) 0 )
	    {
	    /* !!! Check the update socket. */
	    }

	(void) pthread_mutex_unlock( &lock );
	}
    }


static void
trim( char* str )
    {
    char* cp;
    int len;

    cp = strchr( str, '#' );
    if ( cp != (char*) 0 )
	*cp = '\0';
    len = strlen( str );
    while ( str[len-1] == '\n' || str[len-1] == '\r' || str[len-1] == ' ' || str[len-1] == '\t' )
	{
	--len;
	str[len] = '\0';
	}
    }


/* The private data struct. */
struct connection_data {
    int action;
    int nheaders;
    };
#define ACTION_UNKNOWN 0
#define ACTION_REJECT 1
#define ACTION_MARK 2
#define ACTION_TEMPFAIL 3


/* black_connect - handle the initial TCP connection
**
** Called at the start of a connection.  Any per-connection data should
** be initialized here.
**
** connhost: The hostname of the client, based on a reverse lookup.
** connaddr: The client's IP address, based on getpeername().
*/
static sfsistat
black_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr )
    {
    struct connection_data* cd;
    struct sockaddr_in* sin;
    char* char_addr;
    octets ip;
    int i;

    update();

    if ( connaddr == (_SOCK_ADDR*) 0 )
	return SMFIS_ACCEPT;	/* can't deal with it */

    cd = (struct connection_data*) malloc( sizeof(struct connection_data) );
    if ( cd == (struct connection_data*) 0 )
	return SMFIS_TEMPFAIL;
    (void) smfi_setpriv( ctx, (void*) cd );
    cd->action = ACTION_UNKNOWN;
    cd->nheaders = 0;

    sin = (struct sockaddr_in*) connaddr;

    char_addr = (char*) &sin->sin_addr.s_addr;
    ip.a = char_addr[0];
    ip.b = char_addr[1];
    ip.c = char_addr[2];
    ip.d = char_addr[3];
    ip.w = 32;

    for ( i = 0; i < n_whitelists; ++i )
	if ( iparray_get( whitelists[i], ip ) > 0 )
	    {
	    if ( loglistname )
		syslog( LOG_INFO, "whitelist %s \"%s\" [%d.%d.%d.%d]", whitelist_files[i], connhost, (int) ip.a, (int) ip.b, (int) ip.c, (int) ip.d );
	    else
		syslog( LOG_INFO, "whitelist \"%s\" [%d.%d.%d.%d]", connhost, (int) ip.a, (int) ip.b, (int) ip.c, (int) ip.d );
	    return SMFIS_ACCEPT;
	    }
    for ( i = 0; i < n_blacklists; ++i )
	if ( iparray_get( blacklists[i], ip ) >= threshold )
	    {
	    if ( loglistname )
		syslog( LOG_INFO, "blacklist %s \"%s\" [%d.%d.%d.%d]", blacklist_files[i], connhost, (int) ip.a, (int) ip.b, (int) ip.c, (int) ip.d );
	    else
		syslog( LOG_INFO, "blacklist \"%s\" [%d.%d.%d.%d]", connhost, (int) ip.a, (int) ip.b, (int) ip.c, (int) ip.d );
	    if ( markonly )
		cd->action = ACTION_MARK;
	    else if ( graylist )
		cd->action = ACTION_TEMPFAIL;
	    else
		cd->action = ACTION_REJECT;
	    return SMFIS_CONTINUE;
	    }

    return SMFIS_ACCEPT;
    }


/* black_helo - handle the HELO command
**
** Called at the start of a connection.
**
** helohost: The string passed to the HELO/EHLO command.
*/
static sfsistat
black_helo( SMFICTX* ctx, char* helohost )
    {
    struct connection_data* cd = (struct connection_data*) smfi_getpriv( ctx );

    /* The reject and temporary failure responses have to happen in the
    ** HELO handler so that we can send back a proper rejection message.
    ** Can't do that from the connect handler.
    */
    if ( cd->action == ACTION_REJECT )
	{
	(void) smfi_setreply( ctx, "554", "5.7.1", rejectmessage );
	return SMFIS_REJECT;
	}
    else if ( cd->action == ACTION_TEMPFAIL )
	{
	(void) smfi_setreply( ctx, "421", "4.3.2", "temporarily blacklisted - please try again later" );
	return SMFIS_TEMPFAIL;
	}

    return SMFIS_CONTINUE;
    }


/* black_header - handle a header line
**
** Called separately for each header line in a message.
**
** name:  Header field name.
** value: Header vield value, including folded whitespace.  The final CRLF
**        is removed.
*/   
static sfsistat
black_header( SMFICTX* ctx, char* name, char* value )
    {
    struct connection_data* cd = (struct connection_data*) smfi_getpriv( ctx );

    if ( markonly )
	if ( strcasecmp( name, HEADER ) == 0 )
	    ++cd->nheaders;

    return SMFIS_CONTINUE;
    }


/* black_eom - handle the end of the message
**
** Called once per message after all body blocks have been processed.
** Any per-message data should be freed both here and in black_abort().
*/
static sfsistat
black_eom( SMFICTX* ctx )
    {
    struct connection_data* cd = (struct connection_data*) smfi_getpriv( ctx );
    int i;
    char buf[500];

    /* Header deleting and adding can only happen in the eom handler. */
    if ( markonly )
	for ( i = cd->nheaders; i >= 1; --i )
	    (void) smfi_chgheader( ctx, HEADER, i, (char*) 0 );
    if ( cd->action == ACTION_MARK )
	{
	(void) snprintf( buf, sizeof(buf), "%s %s %s", BLACKMILTER_PROGRAM, BLACKMILTER_VERSION, BLACKMILTER_URL );
	(void) smfi_addheader( ctx, HEADER, buf );
	}

    return SMFIS_CONTINUE;
    }


/* black_close - handle the connection being closed
**
** Called once at the end of a connection.  Any per-connection data
** should be freed here.
*/
static sfsistat
black_close( SMFICTX* ctx )
    {
    struct connection_data* cd = (struct connection_data*) smfi_getpriv( ctx );

    if ( cd != (struct connection_data*) 0 )
	{
	(void) smfi_setpriv( ctx, (void*) 0 );
	free( (void*) cd );
	}

    return SMFIS_CONTINUE;
    }
