/* acme_chat - interface to ACME News Chat
**
** Copyright  1998 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
**
** This is an alternative user interface for ACME News's chat and
** instant-message system.  It talks to the back end via a protocol built
** on top of standard authenticated HTTP POSTs.  The protocol is as follows:
**
**   request=CONNECT
**   The client lets the server know it is present.
**   response=OK
**   userid=<name>
**   or
**   response=ERR
**
**   request=SEND_MESSAGE
**   text=<text>
**   to=<userid>    (optional)
**   The client sends a message.
**   response=OK
**   or
**   response=ERR
**
**   request=NEXT_MESSAGE
**   The client requests the next message be returned.
**   response=OK
**   date=<Unix date>
**   from=<userid>
**   to=<userid>    (if it's a private message)
**   text=<text>
**   or
**   response=NOOP
**   or
**   response=ERR
**
**   request=WHOS_ON
**   The client requests a list of all users currently connected.  The
**   server adds the list to the user's pending messages, where it
**   gets retrieved by subsequent NEXT_MESSAGE requests.
**   response=OK
**   or
**   response=ERR
**
**   request=DISCONNECT
**   The client has left the building.
**   response=OK
**   or
**   response=ERR
**
** One subtlety not apparent from the above is that a NEXT_MESSAGE request
** may not get an immediate response.  If there are no messages available,
** the server will wait around for a few minutes to see if any new messages
** show up.  If one does, it gets sent back to fulfill the NEXT_MESSAGE
** request; if the timeout expires, a NOOP response is returned.
*/

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <curses.h>
#include <sys/time.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>


/* Definitions. */

#define NAME_MAX 16

#define PRM_REQUEST "request"
#define PRM_RESPONSE "response"
#define PRM_TEXT "text"
#define PRM_DATE "date"
#define PRM_FROM "from"
#define PRM_TO "to"
#define PRM_USERID "userid"
#define PRM_COMMENT "comment"

#define REQ_CONNECT "CONNECT"
#define REQ_SEND_MESSAGE "SEND_MESSAGE"
#define REQ_NEXT_MESSAGE "NEXT_MESSAGE"
#define REQ_WHOS_ON "WHOS_ON"
#define REQ_DISCONNECT "DISCONNECT"

#define RES_OK "OK"
#define RES_ERR "ERR"
#define RES_NOOP "NOOP"


/* Forwards. */
static void usage( void );
static void draw_sep( void );
static int handle_typein( void );
static int handle_line( void );
static void handle_response( int resp_fd );
static void add_line( char* cp );
static void add_line_int( char* cp, int standout );
static void add_error( char* cp );
static long idle( void );
static void not_idle( void );
static void netinit( char* url );
static int send_request( char* request );
static void lookup_address( char* hostname, short port );
static int open_client_socket( void );
static char* auth_cookie( void );
static void write_str( int fd, char* str );
static int read_response( int fd );
static int hexit( char c );
static void url_decode( char* str );
static void url_encode( const char* from, char* to );
static void nv_clear( void );
static void nv_put( char* name, char* value );
static char* nv_get( char* name );
static int nv_count( void );
static char* nv_nameAt( int i );
static char* nv_valueAt( int i );
static int b64_encode( unsigned char* ptr, int len, char* space, int size );


/* Globals. */

static char* argv0;
static char* url;
static char host[2000];
static int port;
static char* base;
static int base_ends_with_slash;
static char username[100];
static char password[100];
static WINDOW* main_win = (WINDOW*) 0;
static WINDOW* sep_win;
static WINDOW* typein_win;
static char line[5000];
static int line_len;


int
main( int argc, char** argv )
    {
    int argn;
    char prompt[NAME_MAX + 3];
    int prompt_len;
    int resp_fd;
    fd_set read_fds;
    struct timeval tv;
    char buf[100];

    argv0 = argv[0];
    argn = 1;
    while ( argn < argc && argv[argn][0] == '-' )
	{
	usage();
	++argn;
	}
    if ( argn != argc - 1 )
	usage();

    url = argv[argn];
    not_idle();
    netinit( url );

    (void) printf( "Logging in to ACME News Chat at %s\n", url );
    (void) printf( "\n" );
    (void) printf( "Username: " );
    (void) fflush( stdout );
    (void) fgets( username, sizeof(username), stdin );
    username[strlen(username)-1] = '\0';
    if ( strlen( username ) > NAME_MAX )
	{
	(void) fprintf( stderr, "Not a valid username.\n" );
	exit( 1 );
	}
    (void) printf( "Password: " );
    (void) fflush( stdout );
    initscr();
    noecho();
    nl();
    (void) fgets( password, sizeof(password), stdin );
    echo();
    endwin();
    password[strlen(password)-1] = '\0';
    (void) printf( "\n" );
    (void) printf( "Connecting as %s...\n", username );
    (void) fflush( stdout );
    nv_clear();
    if ( ! read_response( send_request( REQ_CONNECT ) ) )
	exit( 1 );

    initscr();
    noecho();
    nonl();
    raw();

    main_win = newwin( LINES - 2, COLS, 0, 0 );
    scrollok( main_win, TRUE );
    /* leaveok( main_win, TRUE ); */
    sep_win = newwin( 1, COLS, LINES - 2, 0 );
    /* leaveok( sep_win, TRUE ); */
    draw_sep();
    typein_win = newwin( 1, COLS, LINES - 1, 0 );

    /* Set up the prompt. */
    (void) sprintf( prompt, "%s> ", username );
    prompt_len = strlen( prompt );

    /* Main loop. */
    line[0] = '\0';
    line_len = 0;
    resp_fd = -1;
    for (;;)
	{
	/* Update the screen. */
	wrefresh( main_win );
	werase( typein_win );
	wmove( typein_win, 0, 0 );
	waddstr( typein_win, prompt );
	if ( line_len <= COLS - prompt_len - 1 )
	    waddstr( typein_win, line );
	else
	    waddstr( typein_win, &line[line_len - ( COLS - prompt_len - 1 )] );
	wrefresh( typein_win );

	/* If necessary, queue a request. */
	if ( resp_fd < 0 )
	    {
	    nv_clear();
	    resp_fd = send_request( REQ_NEXT_MESSAGE );
	    }

	/* Check which fds are readable. */
	FD_ZERO( &read_fds );
	FD_SET( 0, &read_fds );
	if ( resp_fd >= 0 )
	    FD_SET( resp_fd, &read_fds );
	tv.tv_sec = 30;
	tv.tv_usec = 0;
	(void) select( 256, &read_fds, (fd_set*) 0, (fd_set*) 0, &tv );
	if ( FD_ISSET( 0, &read_fds ) )
	    {
	    /* The user has typed a character. */
	    if ( ! handle_typein() )
		break;
	    }
	else if ( resp_fd != -1 && FD_ISSET( resp_fd, &read_fds ) )
	    {
	    /* The HTTP request has completed. */
	    handle_response( resp_fd );
	    resp_fd = -1;
	    }
	else
	    {
	    /* Must be a timeout. */
	    draw_sep();
	    }
	}

    noraw();
    echo();
    endwin();
    main_win = (WINDOW*) 0;

    nv_clear();
    (void) read_response( send_request( REQ_DISCONNECT ) );

    exit( 0 );
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage:  %s URL\n", argv0 );
    exit( 1 );
    }


static void
draw_sep( void )
    {
    int i;
    char* title = "ACME-News-Chat";
    time_t t;
    struct tm* tmP;
    char buf[10];

    wstandout( sep_win );
    for ( i = 0; i < COLS; ++i )
	waddch( sep_win, '-' );
    wmove( sep_win, 0, ( COLS - strlen( title ) ) / 2 );
    waddstr( sep_win, title );
    t = time( (time_t*) 0 );
    tmP = localtime( &t );
    (void) sprintf( buf, "%02d:%02d", tmP->tm_hour, tmP->tm_min );
    wmove( sep_win, 0, COLS - 5 );
    waddstr( sep_win, buf );
    wstandend( sep_win );
    wrefresh( sep_win );
    }


static int
handle_typein( void )
    {
    int ch;

    ch = wgetch( typein_win );
    if ( ( ch == '\003' || ch == '\004' ) && line_len == 0 )
	return 0;
    else if ( ch == '\n' || ch == '\r' )
	return handle_line();
    else if ( ch == '\b' || ch == '\177' )
	{
	if ( line_len > 0 )
	    --line_len;
	}
    else if ( ch == '\025' )	/* ^U */
	line_len = 0;
    else if ( ch == '\027' )	/* ^W */
	{
	while ( line_len > 0 && line[line_len - 1] == ' ' )
	    --line_len;
	while ( line_len > 0 && line[line_len - 1] != ' ' )
	    --line_len;
	}
    else if ( ch == '\014' )	/* ^L */
	{
	touchwin( curscr );
	wrefresh( curscr );
	}
    else if ( ch == '\032' )	/* ^Z */
	{
	noraw();
	echo();
	(void) kill( 0, SIGTSTP );
	noecho();
	raw();
	touchwin( curscr );
	draw_sep();
	wrefresh( curscr );
	}
    else if ( isprint( ch ) && line_len + 1 < sizeof(line) )
	line[line_len++] = ch;
    line[line_len] = '\0';
    return 1;
    }


static int
handle_line( void )
    {
    not_idle();

    if ( line_len == 0 )
	return;		/* ignore blank lines */

    if ( line[0] == '/' )
	{
	if ( strcasecmp( line, "/quit" ) == 0 )
	    return 0;
	else if ( strcasecmp( line, "/who" ) == 0 )
	    {
	    nv_clear();
	    (void) read_response( send_request( REQ_WHOS_ON ) );
	    }
	else if ( strncasecmp( line, "/send", 5 ) == 0 )
	    {
	    char to[100], text[5000];
	    if ( sscanf( line, "/send %[^ ] %[^\n\r]", to, text ) != 2 )
		add_error( "Usage: /send username message" );
	    else
		{
		nv_clear();
		nv_put( PRM_TO, to );
		nv_put( PRM_TEXT, text );
		(void) read_response( send_request( REQ_SEND_MESSAGE ) );
		}
	    }
	else if ( strcasecmp( line, "/help" ) == 0 )
	    add_error( "The only commands implemented at the moment: /who /send /quit /help" );
	else
	    {
	    char buf[100];
	    (void) sprintf( buf, "Unrecognized command: '%.50s'", line );
	    add_error( buf );
	    add_error( "/help for command list" );
	    }
	}
    else
	{
	nv_clear();
	nv_put( PRM_TEXT, line );
	(void) read_response( send_request( REQ_SEND_MESSAGE ) );
	}

    /* And clear out the typein line. */
    line_len = 0;
    line[line_len] = '\0';
    return 1;
    }


static void
handle_response( int resp_fd )
    {
    char* response;
    char* text;
    char* date;
    char* from;
    char* to;
    char* comment;
    char buf[5000];
    time_t t;
    struct tm* tmP;
    char tbuf[100];

    if ( ! read_response( resp_fd ) )
	return;
    response = nv_get( PRM_RESPONSE );
    if ( response == (char*) 0 )
	add_error( "Bogus response from server." );
    else if ( strcmp( response, RES_OK ) == 0 )
	{
	text = nv_get( PRM_TEXT );
	if ( text == (char*) 0 )
	    add_error( "Missing text in response." );
	else
	    {
	    while ( text[strlen(text)-1] == '\n' )
		text[strlen(text)-1] = '\0';
	    date = nv_get( PRM_DATE );
	    from = nv_get( PRM_FROM );
	    to = nv_get( PRM_TO );
	    buf[0] = '\0';
	    if ( to != (char*) 0 )
		(void) strcat( buf, "[" );
	    if ( date != (char*) 0 )
		{
		t = atol( date );
		tmP = localtime( &t );
		(void) sprintf( tbuf, "%d:%02d", tmP->tm_hour, tmP->tm_min );
		(void) strcat( buf, tbuf );
		}
	    if ( from != (char*) 0 )
		{
		(void) strcat( buf, " " );
		(void) strcat( buf, from );
		}
	    if ( to != (char*) 0 )
		{
		(void) strcat( buf, " to " );
		(void) strcat( buf, to );
		}
	    if ( date != (char*) 0 || from != (char*) 0 || to != (char*) 0 )
		(void) strcat( buf, ": " );
	    (void) strcat( buf, text );
	    if ( to != (char*) 0 )
		(void) strcat( buf, "]" );
	    if ( to != (char*) 0 )
		add_line_int( buf, 1 );
	    else
		add_line( buf );
	    }
	}
    else if ( strcmp( response, RES_NOOP ) == 0 )
	{ /* ignore */ }
    else if ( strcmp( response, RES_ERR ) == 0 )
	{
	comment = nv_get( PRM_COMMENT );
	(void) sprintf( buf, "Error response from server - %s", comment );
	add_error( buf );
	}
    }


static int beep_seconds = 10 * 60;


static void
add_line( char* cp )
    {
    add_line_int( cp, 0 );
    }


static void
add_line_int( char* cp, int standout )
    {
    static char bel = '\007';

    if ( idle() >= beep_seconds )
	(void) write( 1, &bel, 1 );

    if ( main_win != (WINDOW*) 0 )
	{
	if ( standout )
	    wstandout( main_win );
	waddch( main_win, '\n' );
	waddstr( main_win, cp );
	if ( standout )
	    wstandend( main_win );
	}
    else
	(void) printf( "%s\n", cp );
    }


static void
add_error( char* cp )
    {
    add_line_int( cp, 1 );
    }


static time_t last_active;


static long
idle( void )
    {
    time_t now = time( (time_t*) 0 );
    long idle = now - last_active;
    last_active = now;
    return idle;
    }


static void
not_idle( void )
    {
    last_active = time( (time_t*) 0 );
    }


static void
netinit( char* url )
    {
    char* s;
    int hostLen;
    char* http = "http://";
    int httpLen = strlen( http );

    if ( strncmp( http, url, httpLen ) )
        {
	(void) fprintf( stderr, "%s: non-HTTP URL\n", argv0 );
        exit( 1 );
        }

    /* Get the host name. */
    for ( s = url + httpLen; *s != '\0' && *s != ':' && *s != '/'; ++s )
	;
    hostLen = s - url;
    hostLen -= httpLen;
    strncpy( host, url + httpLen, hostLen );
    host[hostLen] = '\0';

    /* Get port number. */
    if ( *s == ':' )
	{
	port = atoi( ++s );
	while ( *s != '\0' && *s != '/' )
	    ++s;
	}
    else
	port = 80;

    /* Get the base file name. */
    if ( *s == '\0' )
	base = "/";
    else
	base = s;
    base_ends_with_slash = ( base[strlen( base ) - 1] == '/' );

    lookup_address( host, port );
    }


static int
send_request( char* request )
    {
    int fd;
    int count, i, reqlen;
    char reqbuf[5000];
    char* name;
    char* value;
    char value_encoded[5000];
    struct sockaddr_in sin;
    char fmtbuf[100];

    /* Format the request. */
    nv_put( PRM_REQUEST, request );
    count = nv_count();
    reqlen = 0;
    for ( i = 0; i < count; ++i )
	{
	if ( reqlen != 0 )
	    reqbuf[reqlen++] = '&';
	name = nv_nameAt( i );
	(void) strcpy( &(reqbuf[reqlen]), name );
	reqlen += strlen( name );
	reqbuf[reqlen++] = '=';
	value = nv_valueAt( i );
	url_encode( value, value_encoded );
	(void) strcpy( &(reqbuf[reqlen]), value_encoded );
	reqlen += strlen( value_encoded );
	}
    reqbuf[reqlen] = '\0';

    /* Open the socket. */
    fd = open_client_socket();
    if ( fd < 0 )
	return -1;

    /* Write the headers. */
    if ( base_ends_with_slash )
	(void) sprintf( fmtbuf, "POST %smessage.cgi HTTP/1.0\n", base );
    else
	(void) sprintf( fmtbuf, "POST %s/message.cgi HTTP/1.0\n", base );
    write_str( fd, fmtbuf );
    (void) sprintf( fmtbuf, "Authorization: Basic %s\n", auth_cookie() );
    write_str( fd, fmtbuf );
    write_str( fd, "Content-type: application/x-www-form-urlencoded\n" );
    (void) sprintf( fmtbuf, "Content-length: %d\n\n", reqlen );
    write_str( fd, fmtbuf );
    /* And write the request data. */
    (void) write( fd, reqbuf, reqlen );

    return fd;
    }


#if defined(AF_INET6) && defined(IN6_IS_ADDR_V4MAPPED)
#define USE_IPV6
#endif

#ifdef USE_IPV6
    struct sockaddr_in6 sa;
#else /* USE_IPV6 */
    struct sockaddr_in sa;
#endif /* USE_IPV6 */
    int sa_len, sock_family, sock_type, sock_protocol;

static void
lookup_address( char* hostname, short port )
    {
#ifdef USE_IPV6
    struct addrinfo hints;
    char portstr[10];
    int gaierr;
    struct addrinfo* ai;
    struct addrinfo* ai2;
    struct addrinfo* aiv4;
    struct addrinfo* aiv6;
#else /* USE_IPV6 */
    struct hostent *he;
#endif /* USE_IPV6 */

    (void) memset( (void*) &sa, 0, sizeof(sa) );

#ifdef USE_IPV6

    (void) memset( &hints, 0, sizeof(hints) );
    hints.ai_family = PF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    (void) snprintf( portstr, sizeof(portstr), "%d", port );
    if ( (gaierr = getaddrinfo( hostname, portstr, &hints, &ai )) != 0 )
	{
	(void) fprintf(
	    stderr, "%s: getaddrinfo %s - %s\n", argv0, hostname,
	    gai_strerror( gaierr ) );
	exit( 1 );
	}

    /* Find the first IPv4 and IPv6 entries. */
    aiv4 = (struct addrinfo*) 0;
    aiv6 = (struct addrinfo*) 0;
    for ( ai2 = ai; ai2 != (struct addrinfo*) 0; ai2 = ai2->ai_next )
	{
	switch ( ai2->ai_family )
	    {
	    case AF_INET:
	    if ( aiv4 == (struct addrinfo*) 0 )
		aiv4 = ai2;
	    break;
	    case AF_INET6:
	    if ( aiv6 == (struct addrinfo*) 0 )
		aiv6 = ai2;
	    break;
	    }
	}

    /* If there's an IPv4 address, use that, otherwise try IPv6. */
    if ( aiv4 != (struct addrinfo*) 0 )
	{
	if ( sizeof(sa) < aiv4->ai_addrlen )
	    {
	    (void) fprintf(
		stderr, "%s - sockaddr too small (%lu < %lu)\n",
		hostname, (unsigned long) sizeof(sa),
		(unsigned long) aiv4->ai_addrlen );
	    exit( 1 );
	    }
	sock_family = aiv4->ai_family;
	sock_type = aiv4->ai_socktype;
	sock_protocol = aiv4->ai_protocol;
	sa_len = aiv4->ai_addrlen;
	(void) memmove( &sa, aiv4->ai_addr, sa_len );
	freeaddrinfo( ai );
	return;
	}
    if ( aiv6 != (struct addrinfo*) 0 )
	{
	if ( sizeof(sa) < aiv6->ai_addrlen )
	    {
	    (void) fprintf(
		stderr, "%s - sockaddr too small (%lu < %lu)\n",
		hostname, (unsigned long) sizeof(sa),
		(unsigned long) aiv6->ai_addrlen );
	    exit( 1 );
	    }
	sock_family = aiv6->ai_family;
	sock_type = aiv6->ai_socktype;
	sock_protocol = aiv6->ai_protocol;
	sa_len = aiv6->ai_addrlen;
	(void) memmove( &sa, aiv6->ai_addr, sa_len );
	freeaddrinfo( ai );
	return;
	}

    (void) fprintf(
	stderr, "%s: no valid address found for host %s\n", argv0, hostname );
    exit( 1 );

#else /* USE_IPV6 */

    he = gethostbyname( hostname );
    if ( he == (struct hostent*) 0 )
	{
	(void) fprintf( stderr, "%s: unknown host - %s\n", argv0, hostname );
	exit( 1 );
	}
    sock_family = sa.sin_family = he->h_addrtype;
    sock_type = SOCK_STREAM;
    sock_protocol = 0;
    sa_len = sizeof(sa);
    (void) memmove( &sa.sin_addr, he->h_addr, he->h_length );
    sa.sin_port = htons( port );

#endif /* USE_IPV6 */

    }


static int
open_client_socket( void )
    {
    int sockfd;

    sockfd = socket( sock_family, sock_type, sock_protocol );
    if ( sockfd < 0 )
	{
	add_error( "Problem creating socket." );
	return -1;
	}

    if ( connect( sockfd, (struct sockaddr*) &sa, sa_len ) < 0 )
	{
	add_error( "Problem connecting socket." );
	return -1;
	}

    return sockfd;
    }


static char*
auth_cookie( void )
    {
    static int inited = 0;
    char buf[500];
    static char cookie[500];

    if ( ! inited )
	{
	inited = 1;
	(void) sprintf( buf, "%s:%s", username, password );
	cookie[b64_encode( buf, strlen( buf ), cookie, sizeof(cookie) )] = '\0';
	}
    return cookie;
    }

static void

write_str( int fd, char* str )
    {
    (void) write( fd, str, strlen( str ) );
    }


static int
read_response( int fd )
    {
    FILE* fp;
    char readbuf[5000];
    char parsbuf1[1000], parsbuf2[1000], parsbuf3[1000];
    char fmtbuf[5000];
    char* content_length_str;
    int content_length;
    char* content;
    char* name;
    char* value;
    char* name_chars =
      "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_-";
    int name_len, value_len;

    if ( fd < 0 )
	return 0;

    fp = fdopen( fd, "r" );

    /* Read status line and headers, and put them in the symbol table. */
    nv_clear();
    if ( fgets( readbuf, sizeof(readbuf), fp ) == (char*) 0 )
	{
	add_error( "Problem reading HTTP response status." );
	(void) fclose( fp );
	return 0;
	}
    if ( sscanf( readbuf, "%[^ ] %[^ ] %[^\n\r]", parsbuf1, parsbuf2, parsbuf3 ) != 3 )
	{
	add_error( "Problem parsing HTTP response status." );
	(void) fclose( fp );
	return 0;
	}
    nv_put( "HTTP_Version", parsbuf1 );
    nv_put( "HTTP_Status", parsbuf2 );
    nv_put( "HTTP_Comment", parsbuf3 );
    for (;;)
	{
	if ( fgets( readbuf, sizeof(readbuf), fp ) == (char*) 0 )
	    {
	    add_error( "Problem reading HTTP response header." );
	    (void) fclose( fp );
	    return 0;
	    }
	if ( ( readbuf[0] == '\n' && readbuf[1] == '\0' ) ||
	     ( readbuf[0] == '\r' && readbuf[1] == '\0' ) ||
	     ( readbuf[0] == '\r' && readbuf[1] == '\n' && readbuf[2] == '\0' ) )
	    break;
	if ( sscanf( readbuf, "%[^: ]: %[^\n\r]", parsbuf1, parsbuf2 ) != 2 )
	    {
	    add_error( "Problem parsing HTTP response header." );
	    (void) fclose( fp );
	    return 0;
	    }
	(void) sprintf( fmtbuf, "HTTP_%s", parsbuf1 );
	nv_put( fmtbuf, parsbuf2 );
	}

    if ( strcmp( nv_get( "HTTP_Status" ), "401" ) == 0 )
	{
	add_error( "Authorization failed." );
	(void) fclose( fp );
	return 0;
	}
    content_length_str = nv_get( "HTTP_Content-length" );
    if ( content_length_str == (char*) 0 )
	{
	add_error( "No Content-length found." );
	(void) fclose( fp );
	return 0;
	}
    content_length = atoi( content_length_str );
    content = malloc( content_length + 1 );
    if ( content == (char*) 0 )
	{
	add_error( "Couldn't malloc response data buffer." );
	(void) fclose( fp );
	return 0;
	}
    if ( fread( content, 1, content_length, fp ) != content_length )
	{
	add_error( "Couldn't read response data." );
	free( (void*) content );
	(void) fclose( fp );
	return 0;
	}
    content[content_length] = '\0';
    /* Split it up into name-value pairs. */
    for ( name = content; name < content + content_length; )
	{
	/* Span the variable name. */
	name_len = strspn( name, name_chars );
	if ( name_len == 0 )
	    {
	    add_error( "Malformed POST data - missing name." );
	    break;
	    }
	value = name + name_len;
	/* Make sure the = is there. */
	if ( *value != '=' )
	    {
	    add_error( "Malformed POST data - missing =." );
	    break;
	    }
	*value = '\0';
	++value;
	/* Span the value. */
	value_len = strcspn( value, "&" );
	/* Make sure the & (or EOS) is there. */
	if ( value[value_len] == '&' )
	    value[value_len] = '\0';
	else if ( value[value_len] != '\0' )
	    {
	    add_error( "Malformed POST data - missing &." );
	    break;
	    }
	url_decode( value );
	/* Save the name and value. */
	nv_put( name, value );
	/* Advance to the next pair. */
	name = value + value_len + 1;
	}
    free( (void*) content );
    (void) fclose( fp );
    return 1;
    }


/* The legal hexadecimal digits. */
static char is_hexit[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 00-1f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0, /* 20-3f */
    0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1, /* 40-5f */
    0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 60-7f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 80-9f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* a0-bf */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* c0-df */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* e0-ff */
    };


/* Return the value of a hexadecimal digit. */
static int
hexit( char c )
    {
    if ( c >= '0' && c <= '9' )
	return c - '0';
    else if ( c >= 'a' && c <= 'f' )
	return c - 'a' + 10;
    else if ( c >= 'A' && c <= 'F' )
	return c - 'A' + 10;
    else
	return -10000;	/* what ever */
    }


/* URL-encoding. */

/* The characters that don't need encoding: a-z A-Z 0-9 -_.* */
static char safe_chars[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 00-1f */
    0,0,0,0,0,0,0,0,0,0,1,0,0,1,1,0,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0, /* 20-3f */
    0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,1, /* 40-5f */
    0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0, /* 60-7f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 80-9f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* a0-bf */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* c0-df */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* e0-ff */
    };


static void
url_encode( const char* from, char* to )
    {
    int d1, d2;
    char* hexits = "0123456789ABCDEF";

    for ( ; *from != '\0'; ++from, ++to )
	{
	if ( *from == ' ' )
	    *to = '+';
	else if ( safe_chars[(unsigned char) *from] )
	    *to = *from;
	else
	    {
	    d1 = ( *from & 0xf0 ) >> 4;
	    d2 = *from & 0x0f;
	    *to++ = '%';
	    *to++ = hexits[d1];
	    *to = hexits[d2];
	    }
	}
    *to = '\0';
    }


/* Decode URL-encoding in-place. */
static void
url_decode( char* str )
    {
    char* from;
    char* to;
    char c1, c2, nc;
    char* legal_chars = "\t\n\r\"!#$%&'()+,/:;<=>?@[\\]^`{|}~";

    for ( from = to = str; *from != '\0'; ++from, ++to )
	{
	if ( *from == '+' )
	    /* Plus turns to space. */
	    *to = ' ';
	else if ( *from != '%' )
	    /* Non-plus non-percent characters are copied unmodified. */
	    *to = *from;
	else
	    {
	    /* A percent. */
	    ++from;
	    c1 = *from;
	    if ( c1 == '\0' )
		{
		/* A percent at the end of the string.  Put it back and quit. */
		*to = '%';
		break;
		}
	    if ( ! is_hexit[c1] )
		{
		/* A percent and a non-hexit.  Put them back and continue. */
		*to = '%';
		++to;
		*to = c1;
		continue;
		}
	    ++from;
	    c2 = *from;
	    if ( c2 == '\0' )
		{
		/* A percent and one character at the end of the string.
		** put them back and quit.
		*/
		*to = '%';
		++to;
		*to = c1;
		break;
		}
	    if ( ! is_hexit[c2] )
		{
		/* A percent, a hexit, and a non-hexit.  Put them all back
		** and continue.
		*/
		*to = '%';
		++to;
		*to = c1;
		++to;
		*to = c2;
		continue;
		}
	    nc = 16 * hexit( c1 ) + hexit( c2 );
	    /* Check if it's a character we're willing to translate. */
	    if ( nc == 0 || strchr( legal_chars, nc ) == (char*) 0 )
		{
		/* Nope.  Put back the characters untranslated and continue. */
		*to = '%';
		++to;
		*to = c1;
		++to;
		*to = c2;
		continue;
		}
	    else
		/* Yes. */
		*to = nc;
	    }
	}
    /* And terminate the decoded string. */
    *to = '\0';
    }


/* A quick&dirty symbol table package. Linear search. No duplicate checking. */

static int nv_num = 0;
static int nv_size = 0;
static char** nv_names;
static char** nv_values;


static void
nv_clear( void )
    {
    int i;

    for ( i = 0; i < nv_num; ++i )
	{
	free( (void*) nv_names[i] );
	free( (void*) nv_values[i] );
	}
    nv_num = 0;
    }


static void
nv_put( char* name, char* value )
    {
    if ( nv_num >= nv_size )
	{
	if ( nv_size == 0 )
	    {
	    nv_size = 100;
	    nv_names = (char**) malloc( nv_size * sizeof(char**) );
	    nv_values = (char**) malloc( nv_size * sizeof(char**) );
	    }
	else
	    {
	    nv_size *= 2;
	    nv_names = (char**) realloc( (void*) nv_names, nv_size * sizeof(char**) );
	    nv_values = (char**) realloc( (void*) nv_values, nv_size * sizeof(char**) );
	    }
	}
    nv_names[nv_num] = strdup( name );
    nv_values[nv_num] = strdup( value );
    ++nv_num;
    }


static char*
nv_get( char* name )
    {
    int i;

    for ( i = 0; i < nv_num; ++i )
	{
	if ( strcasecmp( name, nv_names[i] ) == 0 )
	    return nv_values[i];
	}
    return (char*) 0;
    }


static int
nv_count( void )
    {
    return nv_num;
    }


static char*
nv_nameAt( int i )
    {
    if ( i < nv_num )
	return nv_names[i];
    else
	return (char*) 0;
    }


static char*
nv_valueAt( int i )
    {
    if ( i < nv_num )
	return nv_values[i];
    else
	return (char*) 0;
    }


/* Base-64 encoding.  This encodes binary data as printable ASCII characters.
** Three 8-bit binary bytes are turned into four 6-bit values, like so:
**
**   [11111111]  [22222222]  [33333333]
**
**   [111111] [112222] [222233] [333333]
**
** Then the 6-bit values are represented using the characters "A-Za-z0-9+/".
*/

static char b64_encode_table[64] = {
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',  /* 0-7 */
    'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',  /* 8-15 */
    'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',  /* 16-23 */
    'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',  /* 24-31 */
    'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',  /* 32-39 */
    'o', 'p', 'q', 'r', 's', 't', 'u', 'v',  /* 40-47 */
    'w', 'x', 'y', 'z', '0', '1', '2', '3',  /* 48-55 */
    '4', '5', '6', '7', '8', '9', '+', '/'   /* 56-63 */
    };

static int b64_decode_table[256] = {
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 00-0F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 10-1F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,62,-1,-1,-1,63,  /* 20-2F */
    52,53,54,55,56,57,58,59,60,61,-1,-1,-1,-1,-1,-1,  /* 30-3F */
    -1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,  /* 40-4F */
    15,16,17,18,19,20,21,22,23,24,25,-1,-1,-1,-1,-1,  /* 50-5F */
    -1,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,  /* 60-6F */
    41,42,43,44,45,46,47,48,49,50,51,-1,-1,-1,-1,-1,  /* 70-7F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 80-8F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 90-9F */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* A0-AF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* B0-BF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* C0-CF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* D0-DF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* E0-EF */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1   /* F0-FF */
    };


/* Do base-64 encoding on a hunk of bytes.   Return the actual number of
** bytes generated.  Base-64 encoding takes up 4/3 the space of the original,
** plus a bit for end-padding.  3/2+5 gives a safe margin.
*/
static int
b64_encode( unsigned char* ptr, int len, char* space, int size )
    {
    int ptr_idx, space_idx, phase;
    char c;

    space_idx = 0;
    phase = 0;
    for ( ptr_idx = 0; ptr_idx < len; ++ptr_idx )
	{
	switch ( phase )
	    {
	    case 0:
	    c = b64_encode_table[ptr[ptr_idx] >> 2];
	    if ( space_idx < size )
		space[space_idx++] = c;
	    c = b64_encode_table[( ptr[ptr_idx] & 0x3 ) << 4];
	    if ( space_idx < size )
		space[space_idx++] = c;
	    ++phase;
	    break;
	    case 1:
	    space[space_idx - 1] =
	      b64_encode_table[
		b64_decode_table[space[space_idx - 1]] |
		( ptr[ptr_idx] >> 4 ) ];
	    c = b64_encode_table[( ptr[ptr_idx] & 0xf ) << 2];
	    if ( space_idx < size )
		space[space_idx++] = c;
	    ++phase;
	    break;
	    case 2:
	    space[space_idx - 1] =
	      b64_encode_table[
		b64_decode_table[space[space_idx - 1]] |
		( ptr[ptr_idx] >> 6 ) ];
	    c = b64_encode_table[ptr[ptr_idx] & 0x3f];
	    if ( space_idx < size )
		space[space_idx++] = c;
	    phase = 0;
	    break;
	    }
	}
    /* Pad with ='s. */
    while ( phase++ < 3 )
	if ( space_idx < size )
	    space[space_idx++] = '=';
    return space_idx;
    }
