// XIP.h - XML Iterative Parsing
//
// An XML parsing method even simpler than SAX.
//
// There are two main types of API for parsing XML documents.  DOM parsers
// read in the whole document and build a parse tree.  If the document is
// large, this obviously takes a lot of cycles and memory before you even
// get started handling semantics.  SAX parsers, on the other hand, interpret
// the document on the fly, giving the pieces to callback routines that
// the application has registered.  However this does require that the
// application keep track of what elements are currently in effect,
// typically via a stack.
//
// The idea here is to parse the document on the fly, like SAX, but the
// only type of data returned is text, i.e. the stuff between all the
// markup elements.  The trick is that the text strings come with a list
// of elements attached (and each element has a (name,value) map of
// attributes).  The parser keeps track of the stack of elements currently
// in effect, instead of making every application do it.  Self-closing
// elements ("<foo bar=bletch /foo>") return an empty string with the
// element list attached.
//
// Because only one type of data is getting returned, we can dispense
// with SAX's callbacks and just return the data from a function call.


// Copyright  2003 by Jef Poskanzer <jef@mail.acme.com>.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//
// For commentary on this license please see http://www.acme.com/license.html


#ifndef _XIP_H_
#define _XIP_H_

#include <sys/types.h>

#include <string>
#include <map>
#include <vector>


class XipException
    {
    public:
    const std::string reason;
    const size_t pos;
    const int line;
    XipException( const std::string& _reason, const size_t _pos, const int _line ) :
	reason( _reason ), pos( _pos ), line( _line )
	{
	}
    };


typedef std::map<std::string, std::string> XipAttributes;
class XipElement
    {
    public:
    int id;
    std::string name;
    XipAttributes attributes;
    };
typedef std::vector<XipElement> XipElementVec;


class XipText
    {
    public:
    size_t startpos;
    int startline;
    bool isCdata;
    std::string text;
    XipElementVec elements;

    // A couple of convenience routines - you can also access elements directly.
    const XipElement& innerElement( void ) const { return elements[elements.size()-1]; }
    const XipElement& outerElement( void ) const { return elements[0]; }
    };


// To make a XIP parser you subclass this and implement the two methods.
class XIP
    {
    public:
    virtual bool hasMoreText( void ) throw ( XipException ) = 0;
    virtual const XipText& getNextText( void ) throw ( XipException ) = 0;
    };

#endif // _XIP_H_
